// Copyright 2019 Google LLC
//
// This source code is licensed under the BSD-style license found in the
// LICENSE file in the root directory of this source tree.

#include <assert.h>
#include <stddef.h>

#include <math.h>

#include <xnnpack/common.h>
#include <xnnpack/math-stubs.h>

#include <fp16/bitcasts.h>


// Table of exp2(k / 2048) values, k = 0..2047
static const uint32_t exp2_k_over_2048_table[2048] = {
  0x3F800000, 0x3F800B18, 0x3F801630, 0x3F80214A,
  0x3F802C64, 0x3F803780, 0x3F80429C, 0x3F804DB9,
  0x3F8058D8, 0x3F8063F7, 0x3F806F17, 0x3F807A39,
  0x3F80855B, 0x3F80907E, 0x3F809BA2, 0x3F80A6C7,
  0x3F80B1ED, 0x3F80BD14, 0x3F80C83C, 0x3F80D365,
  0x3F80DE8F, 0x3F80E9BA, 0x3F80F4E6, 0x3F810013,
  0x3F810B41, 0x3F81166F, 0x3F81219F, 0x3F812CD0,
  0x3F813802, 0x3F814334, 0x3F814E68, 0x3F81599C,
  0x3F8164D2, 0x3F817008, 0x3F817B40, 0x3F818678,
  0x3F8191B2, 0x3F819CEC, 0x3F81A828, 0x3F81B364,
  0x3F81BEA1, 0x3F81C9E0, 0x3F81D51F, 0x3F81E05F,
  0x3F81EBA1, 0x3F81F6E3, 0x3F820226, 0x3F820D6A,
  0x3F8218AF, 0x3F8223F5, 0x3F822F3C, 0x3F823A85,
  0x3F8245CE, 0x3F825118, 0x3F825C63, 0x3F8267AF,
  0x3F8272FC, 0x3F827E4A, 0x3F828998, 0x3F8294E8,
  0x3F82A039, 0x3F82AB8B, 0x3F82B6DE, 0x3F82C232,
  0x3F82CD87, 0x3F82D8DC, 0x3F82E433, 0x3F82EF8B,
  0x3F82FAE4, 0x3F83063D, 0x3F831198, 0x3F831CF4,
  0x3F832850, 0x3F8333AE, 0x3F833F0D, 0x3F834A6C,
  0x3F8355CD, 0x3F83612F, 0x3F836C91, 0x3F8377F5,
  0x3F838359, 0x3F838EBF, 0x3F839A25, 0x3F83A58D,
  0x3F83B0F5, 0x3F83BC5F, 0x3F83C7C9, 0x3F83D335,
  0x3F83DEA1, 0x3F83EA0F, 0x3F83F57D, 0x3F8400ED,
  0x3F840C5D, 0x3F8417CF, 0x3F842341, 0x3F842EB4,
  0x3F843A29, 0x3F84459E, 0x3F845115, 0x3F845C8C,
  0x3F846804, 0x3F84737E, 0x3F847EF8, 0x3F848A73,
  0x3F8495F0, 0x3F84A16D, 0x3F84ACEB, 0x3F84B86B,
  0x3F84C3EB, 0x3F84CF6C, 0x3F84DAEF, 0x3F84E672,
  0x3F84F1F6, 0x3F84FD7C, 0x3F850902, 0x3F851489,
  0x3F852012, 0x3F852B9B, 0x3F853725, 0x3F8542B1,
  0x3F854E3D, 0x3F8559CA, 0x3F856558, 0x3F8570E8,
  0x3F857C78, 0x3F858809, 0x3F85939C, 0x3F859F2F,
  0x3F85AAC3, 0x3F85B659, 0x3F85C1EF, 0x3F85CD86,
  0x3F85D91F, 0x3F85E4B8, 0x3F85F052, 0x3F85FBEE,
  0x3F86078A, 0x3F861328, 0x3F861EC6, 0x3F862A65,
  0x3F863606, 0x3F8641A7, 0x3F864D4A, 0x3F8658ED,
  0x3F866491, 0x3F867037, 0x3F867BDD, 0x3F868785,
  0x3F86932D, 0x3F869ED7, 0x3F86AA81, 0x3F86B62D,
  0x3F86C1D9, 0x3F86CD87, 0x3F86D935, 0x3F86E4E5,
  0x3F86F095, 0x3F86FC47, 0x3F8707F9, 0x3F8713AD,
  0x3F871F62, 0x3F872B17, 0x3F8736CE, 0x3F874285,
  0x3F874E3E, 0x3F8759F8, 0x3F8765B3, 0x3F87716E,
  0x3F877D2B, 0x3F8788E9, 0x3F8794A8, 0x3F87A067,
  0x3F87AC28, 0x3F87B7EA, 0x3F87C3AD, 0x3F87CF71,
  0x3F87DB35, 0x3F87E6FB, 0x3F87F2C2, 0x3F87FE8A,
  0x3F880A53, 0x3F88161D, 0x3F8821E8, 0x3F882DB4,
  0x3F883981, 0x3F88454F, 0x3F88511E, 0x3F885CEF,
  0x3F8868C0, 0x3F887492, 0x3F888065, 0x3F888C39,
  0x3F88980F, 0x3F88A3E5, 0x3F88AFBC, 0x3F88BB94,
  0x3F88C76E, 0x3F88D348, 0x3F88DF23, 0x3F88EB00,
  0x3F88F6DD, 0x3F8902BC, 0x3F890E9B, 0x3F891A7C,
  0x3F89265D, 0x3F893240, 0x3F893E24, 0x3F894A08,
  0x3F8955EE, 0x3F8961D5, 0x3F896DBC, 0x3F8979A5,
  0x3F89858F, 0x3F89917A, 0x3F899D66, 0x3F89A953,
  0x3F89B541, 0x3F89C130, 0x3F89CD20, 0x3F89D911,
  0x3F89E503, 0x3F89F0F6, 0x3F89FCEA, 0x3F8A08DF,
  0x3F8A14D5, 0x3F8A20CD, 0x3F8A2CC5, 0x3F8A38BE,
  0x3F8A44B9, 0x3F8A50B4, 0x3F8A5CB1, 0x3F8A68AE,
  0x3F8A74AD, 0x3F8A80AC, 0x3F8A8CAD, 0x3F8A98AE,
  0x3F8AA4B1, 0x3F8AB0B5, 0x3F8ABCBA, 0x3F8AC8BF,
  0x3F8AD4C6, 0x3F8AE0CE, 0x3F8AECD7, 0x3F8AF8E1,
  0x3F8B04EC, 0x3F8B10F8, 0x3F8B1D05, 0x3F8B2913,
  0x3F8B3523, 0x3F8B4133, 0x3F8B4D44, 0x3F8B5957,
  0x3F8B656A, 0x3F8B717E, 0x3F8B7D94, 0x3F8B89AA,
  0x3F8B95C2, 0x3F8BA1DB, 0x3F8BADF4, 0x3F8BBA0F,
  0x3F8BC62B, 0x3F8BD247, 0x3F8BDE65, 0x3F8BEA84,
  0x3F8BF6A4, 0x3F8C02C5, 0x3F8C0EE7, 0x3F8C1B0A,
  0x3F8C272F, 0x3F8C3354, 0x3F8C3F7A, 0x3F8C4BA1,
  0x3F8C57CA, 0x3F8C63F3, 0x3F8C701E, 0x3F8C7C49,
  0x3F8C8876, 0x3F8C94A3, 0x3F8CA0D2, 0x3F8CAD02,
  0x3F8CB933, 0x3F8CC565, 0x3F8CD198, 0x3F8CDDCC,
  0x3F8CEA01, 0x3F8CF637, 0x3F8D026E, 0x3F8D0EA6,
  0x3F8D1ADF, 0x3F8D271A, 0x3F8D3355, 0x3F8D3F92,
  0x3F8D4BCF, 0x3F8D580E, 0x3F8D644D, 0x3F8D708E,
  0x3F8D7CD0, 0x3F8D8913, 0x3F8D9556, 0x3F8DA19B,
  0x3F8DADE1, 0x3F8DBA29, 0x3F8DC671, 0x3F8DD2BA,
  0x3F8DDF04, 0x3F8DEB4F, 0x3F8DF79C, 0x3F8E03E9,
  0x3F8E1038, 0x3F8E1C87, 0x3F8E28D8, 0x3F8E352A,
  0x3F8E417D, 0x3F8E4DD1, 0x3F8E5A25, 0x3F8E667B,
  0x3F8E72D3, 0x3F8E7F2B, 0x3F8E8B84, 0x3F8E97DE,
  0x3F8EA43A, 0x3F8EB096, 0x3F8EBCF3, 0x3F8EC952,
  0x3F8ED5B2, 0x3F8EE212, 0x3F8EEE74, 0x3F8EFAD7,
  0x3F8F073B, 0x3F8F13A0, 0x3F8F2006, 0x3F8F2C6D,
  0x3F8F38D5, 0x3F8F453F, 0x3F8F51A9, 0x3F8F5E15,
  0x3F8F6A81, 0x3F8F76EF, 0x3F8F835D, 0x3F8F8FCD,
  0x3F8F9C3E, 0x3F8FA8B0, 0x3F8FB523, 0x3F8FC197,
  0x3F8FCE0C, 0x3F8FDA82, 0x3F8FE6FA, 0x3F8FF372,
  0x3F8FFFEC, 0x3F900C66, 0x3F9018E2, 0x3F90255E,
  0x3F9031DC, 0x3F903E5B, 0x3F904ADB, 0x3F90575C,
  0x3F9063DE, 0x3F907062, 0x3F907CE6, 0x3F90896B,
  0x3F9095F2, 0x3F90A279, 0x3F90AF02, 0x3F90BB8C,
  0x3F90C816, 0x3F90D4A2, 0x3F90E12F, 0x3F90EDBD,
  0x3F90FA4D, 0x3F9106DD, 0x3F91136E, 0x3F912001,
  0x3F912C94, 0x3F913929, 0x3F9145BE, 0x3F915255,
  0x3F915EED, 0x3F916B86, 0x3F917820, 0x3F9184BB,
  0x3F919158, 0x3F919DF5, 0x3F91AA93, 0x3F91B733,
  0x3F91C3D3, 0x3F91D075, 0x3F91DD18, 0x3F91E9BC,
  0x3F91F661, 0x3F920307, 0x3F920FAE, 0x3F921C56,
  0x3F922900, 0x3F9235AA, 0x3F924256, 0x3F924F03,
  0x3F925BB0, 0x3F92685F, 0x3F92750F, 0x3F9281C0,
  0x3F928E72, 0x3F929B26, 0x3F92A7DA, 0x3F92B490,
  0x3F92C146, 0x3F92CDFE, 0x3F92DAB7, 0x3F92E771,
  0x3F92F42C, 0x3F9300E8, 0x3F930DA5, 0x3F931A63,
  0x3F932723, 0x3F9333E3, 0x3F9340A5, 0x3F934D67,
  0x3F935A2B, 0x3F9366F0, 0x3F9373B6, 0x3F93807D,
  0x3F938D46, 0x3F939A0F, 0x3F93A6D9, 0x3F93B3A5,
  0x3F93C072, 0x3F93CD3F, 0x3F93DA0E, 0x3F93E6DE,
  0x3F93F3AF, 0x3F940082, 0x3F940D55, 0x3F941A29,
  0x3F9426FF, 0x3F9433D6, 0x3F9440AE, 0x3F944D86,
  0x3F945A60, 0x3F94673C, 0x3F947418, 0x3F9480F5,
  0x3F948DD4, 0x3F949AB3, 0x3F94A794, 0x3F94B476,
  0x3F94C159, 0x3F94CE3D, 0x3F94DB22, 0x3F94E808,
  0x3F94F4F0, 0x3F9501D8, 0x3F950EC2, 0x3F951BAD,
  0x3F952898, 0x3F953585, 0x3F954274, 0x3F954F63,
  0x3F955C53, 0x3F956945, 0x3F957637, 0x3F95832B,
  0x3F959020, 0x3F959D16, 0x3F95AA0D, 0x3F95B705,
  0x3F95C3FF, 0x3F95D0F9, 0x3F95DDF5, 0x3F95EAF1,
  0x3F95F7EF, 0x3F9604EE, 0x3F9611EE, 0x3F961EEF,
  0x3F962BF2, 0x3F9638F5, 0x3F9645FA, 0x3F965300,
  0x3F966006, 0x3F966D0E, 0x3F967A18, 0x3F968722,
  0x3F96942D, 0x3F96A13A, 0x3F96AE47, 0x3F96BB56,
  0x3F96C866, 0x3F96D577, 0x3F96E289, 0x3F96EF9D,
  0x3F96FCB1, 0x3F9709C7, 0x3F9716DD, 0x3F9723F5,
  0x3F97310E, 0x3F973E28, 0x3F974B43, 0x3F975860,
  0x3F97657D, 0x3F97729C, 0x3F977FBC, 0x3F978CDD,
  0x3F9799FF, 0x3F97A722, 0x3F97B446, 0x3F97C16C,
  0x3F97CE92, 0x3F97DBBA, 0x3F97E8E3, 0x3F97F60D,
  0x3F980338, 0x3F981065, 0x3F981D92, 0x3F982AC1,
  0x3F9837F0, 0x3F984521, 0x3F985253, 0x3F985F86,
  0x3F986CBB, 0x3F9879F0, 0x3F988727, 0x3F98945F,
  0x3F98A197, 0x3F98AED1, 0x3F98BC0D, 0x3F98C949,
  0x3F98D686, 0x3F98E3C5, 0x3F98F105, 0x3F98FE46,
  0x3F990B88, 0x3F9918CB, 0x3F99260F, 0x3F993355,
  0x3F99409C, 0x3F994DE3, 0x3F995B2C, 0x3F996877,
  0x3F9975C2, 0x3F99830E, 0x3F99905C, 0x3F999DAB,
  0x3F99AAFA, 0x3F99B84C, 0x3F99C59E, 0x3F99D2F1,
  0x3F99E046, 0x3F99ED9B, 0x3F99FAF2, 0x3F9A084A,
  0x3F9A15A3, 0x3F9A22FD, 0x3F9A3059, 0x3F9A3DB5,
  0x3F9A4B13, 0x3F9A5872, 0x3F9A65D2, 0x3F9A7333,
  0x3F9A8096, 0x3F9A8DF9, 0x3F9A9B5E, 0x3F9AA8C4,
  0x3F9AB62B, 0x3F9AC393, 0x3F9AD0FD, 0x3F9ADE67,
  0x3F9AEBD3, 0x3F9AF940, 0x3F9B06AE, 0x3F9B141D,
  0x3F9B218D, 0x3F9B2EFF, 0x3F9B3C71, 0x3F9B49E5,
  0x3F9B575A, 0x3F9B64D0, 0x3F9B7248, 0x3F9B7FC0,
  0x3F9B8D3A, 0x3F9B9AB5, 0x3F9BA831, 0x3F9BB5AE,
  0x3F9BC32C, 0x3F9BD0AC, 0x3F9BDE2C, 0x3F9BEBAE,
  0x3F9BF931, 0x3F9C06B5, 0x3F9C143B, 0x3F9C21C1,
  0x3F9C2F49, 0x3F9C3CD2, 0x3F9C4A5C, 0x3F9C57E7,
  0x3F9C6573, 0x3F9C7301, 0x3F9C8090, 0x3F9C8E20,
  0x3F9C9BB1, 0x3F9CA943, 0x3F9CB6D6, 0x3F9CC46B,
  0x3F9CD201, 0x3F9CDF98, 0x3F9CED30, 0x3F9CFAC9,
  0x3F9D0864, 0x3F9D1600, 0x3F9D239C, 0x3F9D313A,
  0x3F9D3EDA, 0x3F9D4C7A, 0x3F9D5A1C, 0x3F9D67BE,
  0x3F9D7562, 0x3F9D8308, 0x3F9D90AE, 0x3F9D9E55,
  0x3F9DABFE, 0x3F9DB9A8, 0x3F9DC753, 0x3F9DD4FF,
  0x3F9DE2AD, 0x3F9DF05B, 0x3F9DFE0B, 0x3F9E0BBC,
  0x3F9E196E, 0x3F9E2721, 0x3F9E34D6, 0x3F9E428C,
  0x3F9E5043, 0x3F9E5DFB, 0x3F9E6BB4, 0x3F9E796E,
  0x3F9E872A, 0x3F9E94E7, 0x3F9EA2A5, 0x3F9EB064,
  0x3F9EBE25, 0x3F9ECBE6, 0x3F9ED9A9, 0x3F9EE76D,
  0x3F9EF532, 0x3F9F02F9, 0x3F9F10C0, 0x3F9F1E89,
  0x3F9F2C53, 0x3F9F3A1E, 0x3F9F47EB, 0x3F9F55B8,
  0x3F9F6387, 0x3F9F7157, 0x3F9F7F28, 0x3F9F8CFA,
  0x3F9F9ACE, 0x3F9FA8A3, 0x3F9FB679, 0x3F9FC450,
  0x3F9FD228, 0x3F9FE002, 0x3F9FEDDC, 0x3F9FFBB8,
  0x3FA00996, 0x3FA01774, 0x3FA02553, 0x3FA03334,
  0x3FA04116, 0x3FA04EF9, 0x3FA05CDE, 0x3FA06AC3,
  0x3FA078AA, 0x3FA08692, 0x3FA0947B, 0x3FA0A265,
  0x3FA0B051, 0x3FA0BE3E, 0x3FA0CC2C, 0x3FA0DA1B,
  0x3FA0E80B, 0x3FA0F5FD, 0x3FA103F0, 0x3FA111E4,
  0x3FA11FD9, 0x3FA12DD0, 0x3FA13BC7, 0x3FA149C0,
  0x3FA157BA, 0x3FA165B6, 0x3FA173B2, 0x3FA181B0,
  0x3FA18FAF, 0x3FA19DAF, 0x3FA1ABB0, 0x3FA1B9B3,
  0x3FA1C7B7, 0x3FA1D5BC, 0x3FA1E3C2, 0x3FA1F1C9,
  0x3FA1FFD2, 0x3FA20DDC, 0x3FA21BE7, 0x3FA229F3,
  0x3FA23801, 0x3FA2460F, 0x3FA2541F, 0x3FA26231,
  0x3FA27043, 0x3FA27E57, 0x3FA28C6B, 0x3FA29A82,
  0x3FA2A899, 0x3FA2B6B1, 0x3FA2C4CB, 0x3FA2D2E6,
  0x3FA2E102, 0x3FA2EF1F, 0x3FA2FD3E, 0x3FA30B5E,
  0x3FA3197F, 0x3FA327A1, 0x3FA335C5, 0x3FA343E9,
  0x3FA3520F, 0x3FA36037, 0x3FA36E5F, 0x3FA37C89,
  0x3FA38AB3, 0x3FA398E0, 0x3FA3A70D, 0x3FA3B53B,
  0x3FA3C36B, 0x3FA3D19C, 0x3FA3DFCE, 0x3FA3EE02,
  0x3FA3FC37, 0x3FA40A6D, 0x3FA418A4, 0x3FA426DC,
  0x3FA43516, 0x3FA44351, 0x3FA4518D, 0x3FA45FCA,
  0x3FA46E08, 0x3FA47C48, 0x3FA48A89, 0x3FA498CC,
  0x3FA4A70F, 0x3FA4B554, 0x3FA4C39A, 0x3FA4D1E1,
  0x3FA4E029, 0x3FA4EE73, 0x3FA4FCBE, 0x3FA50B0A,
  0x3FA51958, 0x3FA527A6, 0x3FA535F6, 0x3FA54447,
  0x3FA55299, 0x3FA560ED, 0x3FA56F42, 0x3FA57D98,
  0x3FA58BEF, 0x3FA59A48, 0x3FA5A8A2, 0x3FA5B6FD,
  0x3FA5C559, 0x3FA5D3B7, 0x3FA5E215, 0x3FA5F075,
  0x3FA5FED7, 0x3FA60D39, 0x3FA61B9D, 0x3FA62A02,
  0x3FA63868, 0x3FA646D0, 0x3FA65538, 0x3FA663A2,
  0x3FA6720E, 0x3FA6807A, 0x3FA68EE8, 0x3FA69D57,
  0x3FA6ABC7, 0x3FA6BA39, 0x3FA6C8AC, 0x3FA6D720,
  0x3FA6E595, 0x3FA6F40B, 0x3FA70283, 0x3FA710FC,
  0x3FA71F76, 0x3FA72DF2, 0x3FA73C6F, 0x3FA74AED,
  0x3FA7596C, 0x3FA767ED, 0x3FA7766E, 0x3FA784F1,
  0x3FA79376, 0x3FA7A1FB, 0x3FA7B082, 0x3FA7BF0A,
  0x3FA7CD94, 0x3FA7DC1E, 0x3FA7EAAA, 0x3FA7F937,
  0x3FA807C6, 0x3FA81655, 0x3FA824E6, 0x3FA83379,
  0x3FA8420C, 0x3FA850A1, 0x3FA85F37, 0x3FA86DCE,
  0x3FA87C66, 0x3FA88B00, 0x3FA8999B, 0x3FA8A838,
  0x3FA8B6D5, 0x3FA8C574, 0x3FA8D414, 0x3FA8E2B5,
  0x3FA8F158, 0x3FA8FFFC, 0x3FA90EA1, 0x3FA91D48,
  0x3FA92BEF, 0x3FA93A98, 0x3FA94942, 0x3FA957EE,
  0x3FA9669B, 0x3FA97549, 0x3FA983F8, 0x3FA992A9,
  0x3FA9A15B, 0x3FA9B00E, 0x3FA9BEC2, 0x3FA9CD78,
  0x3FA9DC2F, 0x3FA9EAE7, 0x3FA9F9A1, 0x3FAA085C,
  0x3FAA1718, 0x3FAA25D5, 0x3FAA3494, 0x3FAA4354,
  0x3FAA5215, 0x3FAA60D7, 0x3FAA6F9B, 0x3FAA7E60,
  0x3FAA8D26, 0x3FAA9BEE, 0x3FAAAAB7, 0x3FAAB981,
  0x3FAAC84C, 0x3FAAD719, 0x3FAAE5E7, 0x3FAAF4B6,
  0x3FAB0387, 0x3FAB1259, 0x3FAB212C, 0x3FAB3000,
  0x3FAB3ED6, 0x3FAB4DAD, 0x3FAB5C85, 0x3FAB6B5F,
  0x3FAB7A3A, 0x3FAB8916, 0x3FAB97F3, 0x3FABA6D2,
  0x3FABB5B2, 0x3FABC493, 0x3FABD376, 0x3FABE25A,
  0x3FABF13F, 0x3FAC0025, 0x3FAC0F0D, 0x3FAC1DF6,
  0x3FAC2CE0, 0x3FAC3BCC, 0x3FAC4AB9, 0x3FAC59A7,
  0x3FAC6897, 0x3FAC7787, 0x3FAC867A, 0x3FAC956D,
  0x3FACA462, 0x3FACB358, 0x3FACC24F, 0x3FACD147,
  0x3FACE041, 0x3FACEF3C, 0x3FACFE39, 0x3FAD0D37,
  0x3FAD1C36, 0x3FAD2B36, 0x3FAD3A38, 0x3FAD493B,
  0x3FAD583F, 0x3FAD6744, 0x3FAD764B, 0x3FAD8553,
  0x3FAD945D, 0x3FADA368, 0x3FADB274, 0x3FADC181,
  0x3FADD090, 0x3FADDFA0, 0x3FADEEB1, 0x3FADFDC4,
  0x3FAE0CD8, 0x3FAE1BED, 0x3FAE2B03, 0x3FAE3A1B,
  0x3FAE4934, 0x3FAE584F, 0x3FAE676B, 0x3FAE7688,
  0x3FAE85A6, 0x3FAE94C6, 0x3FAEA3E7, 0x3FAEB309,
  0x3FAEC22D, 0x3FAED151, 0x3FAEE078, 0x3FAEEF9F,
  0x3FAEFEC8, 0x3FAF0DF2, 0x3FAF1D1E, 0x3FAF2C4B,
  0x3FAF3B79, 0x3FAF4AA8, 0x3FAF59D9, 0x3FAF690B,
  0x3FAF783E, 0x3FAF8773, 0x3FAF96A9, 0x3FAFA5E0,
  0x3FAFB519, 0x3FAFC453, 0x3FAFD38E, 0x3FAFE2CB,
  0x3FAFF209, 0x3FB00148, 0x3FB01089, 0x3FB01FCB,
  0x3FB02F0E, 0x3FB03E52, 0x3FB04D98, 0x3FB05CDF,
  0x3FB06C28, 0x3FB07B72, 0x3FB08ABD, 0x3FB09A09,
  0x3FB0A957, 0x3FB0B8A6, 0x3FB0C7F7, 0x3FB0D749,
  0x3FB0E69C, 0x3FB0F5F0, 0x3FB10546, 0x3FB1149D,
  0x3FB123F6, 0x3FB1334F, 0x3FB142AA, 0x3FB15207,
  0x3FB16165, 0x3FB170C4, 0x3FB18024, 0x3FB18F86,
  0x3FB19EE9, 0x3FB1AE4D, 0x3FB1BDB3, 0x3FB1CD1A,
  0x3FB1DC83, 0x3FB1EBEC, 0x3FB1FB57, 0x3FB20AC4,
  0x3FB21A32, 0x3FB229A1, 0x3FB23911, 0x3FB24883,
  0x3FB257F6, 0x3FB2676B, 0x3FB276E0, 0x3FB28657,
  0x3FB295D0, 0x3FB2A54A, 0x3FB2B4C5, 0x3FB2C441,
  0x3FB2D3BF, 0x3FB2E33E, 0x3FB2F2BF, 0x3FB30241,
  0x3FB311C4, 0x3FB32149, 0x3FB330CF, 0x3FB34056,
  0x3FB34FDE, 0x3FB35F68, 0x3FB36EF4, 0x3FB37E80,
  0x3FB38E0E, 0x3FB39D9E, 0x3FB3AD2E, 0x3FB3BCC0,
  0x3FB3CC54, 0x3FB3DBE8, 0x3FB3EB7E, 0x3FB3FB16,
  0x3FB40AAF, 0x3FB41A49, 0x3FB429E4, 0x3FB43981,
  0x3FB4491F, 0x3FB458BF, 0x3FB46860, 0x3FB47802,
  0x3FB487A6, 0x3FB4974B, 0x3FB4A6F1, 0x3FB4B698,
  0x3FB4C642, 0x3FB4D5EC, 0x3FB4E598, 0x3FB4F545,
  0x3FB504F3, 0x3FB514A3, 0x3FB52454, 0x3FB53407,
  0x3FB543BB, 0x3FB55370, 0x3FB56327, 0x3FB572DE,
  0x3FB58298, 0x3FB59253, 0x3FB5A20F, 0x3FB5B1CC,
  0x3FB5C18B, 0x3FB5D14B, 0x3FB5E10D, 0x3FB5F0CF,
  0x3FB60094, 0x3FB61059, 0x3FB62020, 0x3FB62FE9,
  0x3FB63FB2, 0x3FB64F7D, 0x3FB65F4A, 0x3FB66F18,
  0x3FB67EE7, 0x3FB68EB8, 0x3FB69E89, 0x3FB6AE5D,
  0x3FB6BE31, 0x3FB6CE07, 0x3FB6DDDF, 0x3FB6EDB8,
  0x3FB6FD92, 0x3FB70D6D, 0x3FB71D4A, 0x3FB72D29,
  0x3FB73D08, 0x3FB74CE9, 0x3FB75CCC, 0x3FB76CB0,
  0x3FB77C95, 0x3FB78C7B, 0x3FB79C63, 0x3FB7AC4D,
  0x3FB7BC37, 0x3FB7CC23, 0x3FB7DC11, 0x3FB7EC00,
  0x3FB7FBF0, 0x3FB80BE1, 0x3FB81BD4, 0x3FB82BC9,
  0x3FB83BBE, 0x3FB84BB6, 0x3FB85BAE, 0x3FB86BA8,
  0x3FB87BA3, 0x3FB88BA0, 0x3FB89B9E, 0x3FB8AB9D,
  0x3FB8BB9E, 0x3FB8CBA0, 0x3FB8DBA4, 0x3FB8EBA9,
  0x3FB8FBAF, 0x3FB90BB7, 0x3FB91BC0, 0x3FB92BCB,
  0x3FB93BD7, 0x3FB94BE4, 0x3FB95BF3, 0x3FB96C03,
  0x3FB97C14, 0x3FB98C27, 0x3FB99C3B, 0x3FB9AC51,
  0x3FB9BC68, 0x3FB9CC81, 0x3FB9DC9A, 0x3FB9ECB6,
  0x3FB9FCD2, 0x3FBA0CF0, 0x3FBA1D10, 0x3FBA2D31,
  0x3FBA3D53, 0x3FBA4D76, 0x3FBA5D9B, 0x3FBA6DC2,
  0x3FBA7DEA, 0x3FBA8E13, 0x3FBA9E3E, 0x3FBAAE6A,
  0x3FBABE97, 0x3FBACEC6, 0x3FBADEF6, 0x3FBAEF28,
  0x3FBAFF5B, 0x3FBB0F8F, 0x3FBB1FC5, 0x3FBB2FFC,
  0x3FBB4035, 0x3FBB506F, 0x3FBB60AA, 0x3FBB70E7,
  0x3FBB8126, 0x3FBB9165, 0x3FBBA1A6, 0x3FBBB1E9,
  0x3FBBC22D, 0x3FBBD272, 0x3FBBE2B9, 0x3FBBF301,
  0x3FBC034A, 0x3FBC1395, 0x3FBC23E2, 0x3FBC3430,
  0x3FBC447F, 0x3FBC54CF, 0x3FBC6521, 0x3FBC7575,
  0x3FBC85CA, 0x3FBC9620, 0x3FBCA678, 0x3FBCB6D1,
  0x3FBCC72B, 0x3FBCD787, 0x3FBCE7E5, 0x3FBCF843,
  0x3FBD08A4, 0x3FBD1905, 0x3FBD2968, 0x3FBD39CD,
  0x3FBD4A33, 0x3FBD5A9A, 0x3FBD6B03, 0x3FBD7B6D,
  0x3FBD8BD8, 0x3FBD9C45, 0x3FBDACB4, 0x3FBDBD24,
  0x3FBDCD95, 0x3FBDDE07, 0x3FBDEE7C, 0x3FBDFEF1,
  0x3FBE0F68, 0x3FBE1FE0, 0x3FBE305A, 0x3FBE40D5,
  0x3FBE5152, 0x3FBE61D0, 0x3FBE7250, 0x3FBE82D1,
  0x3FBE9353, 0x3FBEA3D7, 0x3FBEB45C, 0x3FBEC4E3,
  0x3FBED56B, 0x3FBEE5F4, 0x3FBEF67F, 0x3FBF070C,
  0x3FBF179A, 0x3FBF2829, 0x3FBF38BA, 0x3FBF494C,
  0x3FBF59DF, 0x3FBF6A74, 0x3FBF7B0B, 0x3FBF8BA3,
  0x3FBF9C3C, 0x3FBFACD7, 0x3FBFBD73, 0x3FBFCE11,
  0x3FBFDEB0, 0x3FBFEF50, 0x3FBFFFF2, 0x3FC01096,
  0x3FC0213B, 0x3FC031E1, 0x3FC04289, 0x3FC05332,
  0x3FC063DC, 0x3FC07489, 0x3FC08536, 0x3FC095E5,
  0x3FC0A695, 0x3FC0B747, 0x3FC0C7FB, 0x3FC0D8AF,
  0x3FC0E966, 0x3FC0FA1D, 0x3FC10AD6, 0x3FC11B91,
  0x3FC12C4D, 0x3FC13D0A, 0x3FC14DC9, 0x3FC15E89,
  0x3FC16F4B, 0x3FC1800F, 0x3FC190D3, 0x3FC1A199,
  0x3FC1B261, 0x3FC1C32A, 0x3FC1D3F5, 0x3FC1E4C1,
  0x3FC1F58E, 0x3FC2065D, 0x3FC2172D, 0x3FC227FF,
  0x3FC238D2, 0x3FC249A7, 0x3FC25A7D, 0x3FC26B55,
  0x3FC27C2E, 0x3FC28D08, 0x3FC29DE4, 0x3FC2AEC2,
  0x3FC2BFA1, 0x3FC2D081, 0x3FC2E163, 0x3FC2F246,
  0x3FC3032B, 0x3FC31411, 0x3FC324F9, 0x3FC335E2,
  0x3FC346CD, 0x3FC357B9, 0x3FC368A7, 0x3FC37996,
  0x3FC38A86, 0x3FC39B78, 0x3FC3AC6B, 0x3FC3BD60,
  0x3FC3CE57, 0x3FC3DF4F, 0x3FC3F048, 0x3FC40143,
  0x3FC4123F, 0x3FC4233D, 0x3FC4343C, 0x3FC4453D,
  0x3FC4563F, 0x3FC46742, 0x3FC47848, 0x3FC4894E,
  0x3FC49A56, 0x3FC4AB60, 0x3FC4BC6B, 0x3FC4CD77,
  0x3FC4DE85, 0x3FC4EF95, 0x3FC500A5, 0x3FC511B8,
  0x3FC522CC, 0x3FC533E1, 0x3FC544F8, 0x3FC55610,
  0x3FC5672A, 0x3FC57845, 0x3FC58962, 0x3FC59A80,
  0x3FC5ABA0, 0x3FC5BCC1, 0x3FC5CDE4, 0x3FC5DF08,
  0x3FC5F02E, 0x3FC60155, 0x3FC6127E, 0x3FC623A8,
  0x3FC634D3, 0x3FC64600, 0x3FC6572F, 0x3FC6685F,
  0x3FC67991, 0x3FC68AC4, 0x3FC69BF8, 0x3FC6AD2E,
  0x3FC6BE66, 0x3FC6CF9F, 0x3FC6E0D9, 0x3FC6F215,
  0x3FC70353, 0x3FC71492, 0x3FC725D2, 0x3FC73714,
  0x3FC74858, 0x3FC7599D, 0x3FC76AE3, 0x3FC77C2B,
  0x3FC78D75, 0x3FC79EC0, 0x3FC7B00C, 0x3FC7C15A,
  0x3FC7D2AA, 0x3FC7E3FB, 0x3FC7F54D, 0x3FC806A1,
  0x3FC817F7, 0x3FC8294D, 0x3FC83AA6, 0x3FC84C00,
  0x3FC85D5B, 0x3FC86EB8, 0x3FC88017, 0x3FC89177,
  0x3FC8A2D8, 0x3FC8B43B, 0x3FC8C5A0, 0x3FC8D706,
  0x3FC8E86D, 0x3FC8F9D6, 0x3FC90B41, 0x3FC91CAD,
  0x3FC92E1B, 0x3FC93F8A, 0x3FC950FA, 0x3FC9626C,
  0x3FC973E0, 0x3FC98555, 0x3FC996CC, 0x3FC9A844,
  0x3FC9B9BE, 0x3FC9CB39, 0x3FC9DCB5, 0x3FC9EE34,
  0x3FC9FFB3, 0x3FCA1135, 0x3FCA22B7, 0x3FCA343C,
  0x3FCA45C1, 0x3FCA5749, 0x3FCA68D1, 0x3FCA7A5C,
  0x3FCA8BE8, 0x3FCA9D75, 0x3FCAAF04, 0x3FCAC094,
  0x3FCAD226, 0x3FCAE3BA, 0x3FCAF54F, 0x3FCB06E5,
  0x3FCB187D, 0x3FCB2A17, 0x3FCB3BB2, 0x3FCB4D4F,
  0x3FCB5EED, 0x3FCB708C, 0x3FCB822E, 0x3FCB93D0,
  0x3FCBA575, 0x3FCBB71A, 0x3FCBC8C2, 0x3FCBDA6B,
  0x3FCBEC15, 0x3FCBFDC1, 0x3FCC0F6E, 0x3FCC211D,
  0x3FCC32CE, 0x3FCC4480, 0x3FCC5633, 0x3FCC67E9,
  0x3FCC799F, 0x3FCC8B57, 0x3FCC9D11, 0x3FCCAECC,
  0x3FCCC089, 0x3FCCD247, 0x3FCCE407, 0x3FCCF5C9,
  0x3FCD078C, 0x3FCD1950, 0x3FCD2B16, 0x3FCD3CDE,
  0x3FCD4EA7, 0x3FCD6071, 0x3FCD723D, 0x3FCD840B,
  0x3FCD95DA, 0x3FCDA7AB, 0x3FCDB97E, 0x3FCDCB51,
  0x3FCDDD27, 0x3FCDEEFE, 0x3FCE00D6, 0x3FCE12B0,
  0x3FCE248C, 0x3FCE3669, 0x3FCE4848, 0x3FCE5A28,
  0x3FCE6C0A, 0x3FCE7DED, 0x3FCE8FD2, 0x3FCEA1B9,
  0x3FCEB3A1, 0x3FCEC58A, 0x3FCED775, 0x3FCEE962,
  0x3FCEFB50, 0x3FCF0D40, 0x3FCF1F31, 0x3FCF3124,
  0x3FCF4319, 0x3FCF550F, 0x3FCF6706, 0x3FCF78FF,
  0x3FCF8AFA, 0x3FCF9CF6, 0x3FCFAEF4, 0x3FCFC0F4,
  0x3FCFD2F4, 0x3FCFE4F7, 0x3FCFF6FB, 0x3FD00900,
  0x3FD01B08, 0x3FD02D10, 0x3FD03F1B, 0x3FD05126,
  0x3FD06334, 0x3FD07543, 0x3FD08753, 0x3FD09965,
  0x3FD0AB79, 0x3FD0BD8E, 0x3FD0CFA5, 0x3FD0E1BD,
  0x3FD0F3D7, 0x3FD105F3, 0x3FD11810, 0x3FD12A2F,
  0x3FD13C4F, 0x3FD14E71, 0x3FD16094, 0x3FD172B9,
  0x3FD184DF, 0x3FD19707, 0x3FD1A931, 0x3FD1BB5C,
  0x3FD1CD89, 0x3FD1DFB7, 0x3FD1F1E7, 0x3FD20419,
  0x3FD2164C, 0x3FD22881, 0x3FD23AB7, 0x3FD24CEF,
  0x3FD25F28, 0x3FD27163, 0x3FD283A0, 0x3FD295DE,
  0x3FD2A81E, 0x3FD2BA5F, 0x3FD2CCA2, 0x3FD2DEE6,
  0x3FD2F12C, 0x3FD30374, 0x3FD315BD, 0x3FD32808,
  0x3FD33A54, 0x3FD34CA2, 0x3FD35EF2, 0x3FD37143,
  0x3FD38396, 0x3FD395EA, 0x3FD3A840, 0x3FD3BA98,
  0x3FD3CCF1, 0x3FD3DF4B, 0x3FD3F1A8, 0x3FD40405,
  0x3FD41665, 0x3FD428C6, 0x3FD43B29, 0x3FD44D8D,
  0x3FD45FF3, 0x3FD4725A, 0x3FD484C3, 0x3FD4972E,
  0x3FD4A99A, 0x3FD4BC08, 0x3FD4CE77, 0x3FD4E0E8,
  0x3FD4F35B, 0x3FD505CF, 0x3FD51845, 0x3FD52ABC,
  0x3FD53D35, 0x3FD54FB0, 0x3FD5622C, 0x3FD574AA,
  0x3FD58729, 0x3FD599AA, 0x3FD5AC2D, 0x3FD5BEB1,
  0x3FD5D137, 0x3FD5E3BE, 0x3FD5F647, 0x3FD608D2,
  0x3FD61B5E, 0x3FD62DEC, 0x3FD6407B, 0x3FD6530C,
  0x3FD6659F, 0x3FD67833, 0x3FD68AC9, 0x3FD69D61,
  0x3FD6AFFA, 0x3FD6C295, 0x3FD6D531, 0x3FD6E7CF,
  0x3FD6FA6E, 0x3FD70D10, 0x3FD71FB2, 0x3FD73257,
  0x3FD744FD, 0x3FD757A4, 0x3FD76A4E, 0x3FD77CF9,
  0x3FD78FA5, 0x3FD7A253, 0x3FD7B503, 0x3FD7C7B4,
  0x3FD7DA67, 0x3FD7ED1C, 0x3FD7FFD2, 0x3FD8128A,
  0x3FD82543, 0x3FD837FE, 0x3FD84ABB, 0x3FD85D79,
  0x3FD87039, 0x3FD882FB, 0x3FD895BE, 0x3FD8A883,
  0x3FD8BB49, 0x3FD8CE11, 0x3FD8E0DB, 0x3FD8F3A6,
  0x3FD90673, 0x3FD91942, 0x3FD92C12, 0x3FD93EE4,
  0x3FD951B8, 0x3FD9648D, 0x3FD97763, 0x3FD98A3C,
  0x3FD99D16, 0x3FD9AFF1, 0x3FD9C2CF, 0x3FD9D5AE,
  0x3FD9E88E, 0x3FD9FB70, 0x3FDA0E54, 0x3FDA213A,
  0x3FDA3421, 0x3FDA4709, 0x3FDA59F4, 0x3FDA6CE0,
  0x3FDA7FCD, 0x3FDA92BD, 0x3FDAA5AE, 0x3FDAB8A0,
  0x3FDACB94, 0x3FDADE8A, 0x3FDAF182, 0x3FDB047B,
  0x3FDB1776, 0x3FDB2A72, 0x3FDB3D70, 0x3FDB5070,
  0x3FDB6371, 0x3FDB7674, 0x3FDB8979, 0x3FDB9C7F,
  0x3FDBAF87, 0x3FDBC291, 0x3FDBD59C, 0x3FDBE8A9,
  0x3FDBFBB8, 0x3FDC0EC8, 0x3FDC21DA, 0x3FDC34ED,
  0x3FDC4802, 0x3FDC5B19, 0x3FDC6E32, 0x3FDC814C,
  0x3FDC9468, 0x3FDCA785, 0x3FDCBAA4, 0x3FDCCDC5,
  0x3FDCE0E7, 0x3FDCF40B, 0x3FDD0731, 0x3FDD1A58,
  0x3FDD2D82, 0x3FDD40AC, 0x3FDD53D9, 0x3FDD6707,
  0x3FDD7A36, 0x3FDD8D68, 0x3FDDA09B, 0x3FDDB3CF,
  0x3FDDC706, 0x3FDDDA3E, 0x3FDDED77, 0x3FDE00B3,
  0x3FDE13F0, 0x3FDE272E, 0x3FDE3A6F, 0x3FDE4DB1,
  0x3FDE60F5, 0x3FDE743A, 0x3FDE8781, 0x3FDE9ACA,
  0x3FDEAE14, 0x3FDEC160, 0x3FDED4AE, 0x3FDEE7FD,
  0x3FDEFB4E, 0x3FDF0EA1, 0x3FDF21F5, 0x3FDF354B,
  0x3FDF48A3, 0x3FDF5BFD, 0x3FDF6F58, 0x3FDF82B4,
  0x3FDF9613, 0x3FDFA973, 0x3FDFBCD5, 0x3FDFD038,
  0x3FDFE39E, 0x3FDFF704, 0x3FE00A6D, 0x3FE01DD7,
  0x3FE03143, 0x3FE044B1, 0x3FE05820, 0x3FE06B91,
  0x3FE07F03, 0x3FE09278, 0x3FE0A5EE, 0x3FE0B966,
  0x3FE0CCDF, 0x3FE0E05A, 0x3FE0F3D7, 0x3FE10755,
  0x3FE11AD5, 0x3FE12E57, 0x3FE141DB, 0x3FE15560,
  0x3FE168E7, 0x3FE17C6F, 0x3FE18FFA, 0x3FE1A386,
  0x3FE1B713, 0x3FE1CAA3, 0x3FE1DE34, 0x3FE1F1C7,
  0x3FE2055B, 0x3FE218F1, 0x3FE22C89, 0x3FE24023,
  0x3FE253BE, 0x3FE2675B, 0x3FE27AF9, 0x3FE28E9A,
  0x3FE2A23C, 0x3FE2B5E0, 0x3FE2C985, 0x3FE2DD2C,
  0x3FE2F0D5, 0x3FE30480, 0x3FE3182C, 0x3FE32BDA,
  0x3FE33F89, 0x3FE3533B, 0x3FE366EE, 0x3FE37AA3,
  0x3FE38E59, 0x3FE3A211, 0x3FE3B5CB, 0x3FE3C987,
  0x3FE3DD44, 0x3FE3F103, 0x3FE404C4, 0x3FE41887,
  0x3FE42C4B, 0x3FE44011, 0x3FE453D8, 0x3FE467A2,
  0x3FE47B6D, 0x3FE48F39, 0x3FE4A308, 0x3FE4B6D8,
  0x3FE4CAAA, 0x3FE4DE7E, 0x3FE4F253, 0x3FE5062A,
  0x3FE51A03, 0x3FE52DDD, 0x3FE541B9, 0x3FE55597,
  0x3FE56977, 0x3FE57D58, 0x3FE5913C, 0x3FE5A520,
  0x3FE5B907, 0x3FE5CCEF, 0x3FE5E0D9, 0x3FE5F4C5,
  0x3FE608B2, 0x3FE61CA2, 0x3FE63092, 0x3FE64485,
  0x3FE65879, 0x3FE66C70, 0x3FE68067, 0x3FE69461,
  0x3FE6A85C, 0x3FE6BC59, 0x3FE6D058, 0x3FE6E458,
  0x3FE6F85B, 0x3FE70C5F, 0x3FE72064, 0x3FE7346C,
  0x3FE74875, 0x3FE75C80, 0x3FE7708C, 0x3FE7849B,
  0x3FE798AB, 0x3FE7ACBD, 0x3FE7C0D0, 0x3FE7D4E6,
  0x3FE7E8FD, 0x3FE7FD15, 0x3FE81130, 0x3FE8254C,
  0x3FE8396A, 0x3FE84D8A, 0x3FE861AC, 0x3FE875CF,
  0x3FE889F4, 0x3FE89E1B, 0x3FE8B243, 0x3FE8C66D,
  0x3FE8DA99, 0x3FE8EEC7, 0x3FE902F7, 0x3FE91728,
  0x3FE92B5B, 0x3FE93F90, 0x3FE953C6, 0x3FE967FE,
  0x3FE97C38, 0x3FE99074, 0x3FE9A4B1, 0x3FE9B8F1,
  0x3FE9CD32, 0x3FE9E175, 0x3FE9F5B9, 0x3FEA09FF,
  0x3FEA1E47, 0x3FEA3291, 0x3FEA46DD, 0x3FEA5B2A,
  0x3FEA6F79, 0x3FEA83CA, 0x3FEA981C, 0x3FEAAC71,
  0x3FEAC0C7, 0x3FEAD51F, 0x3FEAE978, 0x3FEAFDD4,
  0x3FEB1231, 0x3FEB2690, 0x3FEB3AF1, 0x3FEB4F53,
  0x3FEB63B7, 0x3FEB781D, 0x3FEB8C85, 0x3FEBA0EF,
  0x3FEBB55A, 0x3FEBC9C7, 0x3FEBDE36, 0x3FEBF2A6,
  0x3FEC0719, 0x3FEC1B8D, 0x3FEC3003, 0x3FEC447A,
  0x3FEC58F4, 0x3FEC6D6F, 0x3FEC81EC, 0x3FEC966B,
  0x3FECAAEC, 0x3FECBF6E, 0x3FECD3F2, 0x3FECE878,
  0x3FECFD00, 0x3FED1189, 0x3FED2614, 0x3FED3AA1,
  0x3FED4F30, 0x3FED63C1, 0x3FED7853, 0x3FED8CE7,
  0x3FEDA17D, 0x3FEDB615, 0x3FEDCAAE, 0x3FEDDF4A,
  0x3FEDF3E7, 0x3FEE0886, 0x3FEE1D26, 0x3FEE31C9,
  0x3FEE466D, 0x3FEE5B13, 0x3FEE6FBB, 0x3FEE8464,
  0x3FEE9910, 0x3FEEADBD, 0x3FEEC26C, 0x3FEED71C,
  0x3FEEEBCF, 0x3FEF0083, 0x3FEF1539, 0x3FEF29F1,
  0x3FEF3EAB, 0x3FEF5367, 0x3FEF6824, 0x3FEF7CE3,
  0x3FEF91A4, 0x3FEFA667, 0x3FEFBB2B, 0x3FEFCFF2,
  0x3FEFE4BA, 0x3FEFF984, 0x3FF00E4F, 0x3FF0231D,
  0x3FF037EC, 0x3FF04CBD, 0x3FF06190, 0x3FF07665,
  0x3FF08B3B, 0x3FF0A014, 0x3FF0B4EE, 0x3FF0C9CA,
  0x3FF0DEA8, 0x3FF0F387, 0x3FF10868, 0x3FF11D4C,
  0x3FF13231, 0x3FF14717, 0x3FF15C00, 0x3FF170EB,
  0x3FF185D7, 0x3FF19AC5, 0x3FF1AFB5, 0x3FF1C4A6,
  0x3FF1D99A, 0x3FF1EE8F, 0x3FF20386, 0x3FF2187F,
  0x3FF22D7A, 0x3FF24277, 0x3FF25775, 0x3FF26C75,
  0x3FF28177, 0x3FF2967B, 0x3FF2AB81, 0x3FF2C088,
  0x3FF2D592, 0x3FF2EA9D, 0x3FF2FFAA, 0x3FF314B9,
  0x3FF329C9, 0x3FF33EDC, 0x3FF353F0, 0x3FF36906,
  0x3FF37E1E, 0x3FF39338, 0x3FF3A853, 0x3FF3BD71,
  0x3FF3D290, 0x3FF3E7B1, 0x3FF3FCD4, 0x3FF411F9,
  0x3FF4271F, 0x3FF43C48, 0x3FF45172, 0x3FF4669E,
  0x3FF47BCC, 0x3FF490FB, 0x3FF4A62D, 0x3FF4BB60,
  0x3FF4D096, 0x3FF4E5CD, 0x3FF4FB06, 0x3FF51040,
  0x3FF5257D, 0x3FF53ABC, 0x3FF54FFC, 0x3FF5653E,
  0x3FF57A82, 0x3FF58FC8, 0x3FF5A50F, 0x3FF5BA59,
  0x3FF5CFA4, 0x3FF5E4F1, 0x3FF5FA40, 0x3FF60F91,
  0x3FF624E4, 0x3FF63A39, 0x3FF64F8F, 0x3FF664E7,
  0x3FF67A41, 0x3FF68F9D, 0x3FF6A4FB, 0x3FF6BA5B,
  0x3FF6CFBC, 0x3FF6E520, 0x3FF6FA85, 0x3FF70FEC,
  0x3FF72555, 0x3FF73AC0, 0x3FF7502D, 0x3FF7659B,
  0x3FF77B0B, 0x3FF7907E, 0x3FF7A5F2, 0x3FF7BB68,
  0x3FF7D0DF, 0x3FF7E659, 0x3FF7FBD5, 0x3FF81152,
  0x3FF826D1, 0x3FF83C52, 0x3FF851D5, 0x3FF8675A,
  0x3FF87CE1, 0x3FF89269, 0x3FF8A7F4, 0x3FF8BD80,
  0x3FF8D30E, 0x3FF8E89E, 0x3FF8FE30, 0x3FF913C4,
  0x3FF9295A, 0x3FF93EF1, 0x3FF9548B, 0x3FF96A26,
  0x3FF97FC3, 0x3FF99562, 0x3FF9AB03, 0x3FF9C0A6,
  0x3FF9D64A, 0x3FF9EBF1, 0x3FFA0199, 0x3FFA1743,
  0x3FFA2CF0, 0x3FFA429E, 0x3FFA584D, 0x3FFA6DFF,
  0x3FFA83B3, 0x3FFA9968, 0x3FFAAF20, 0x3FFAC4D9,
  0x3FFADA94, 0x3FFAF051, 0x3FFB0610, 0x3FFB1BD1,
  0x3FFB3194, 0x3FFB4758, 0x3FFB5D1F, 0x3FFB72E7,
  0x3FFB88B2, 0x3FFB9E7E, 0x3FFBB44C, 0x3FFBCA1C,
  0x3FFBDFED, 0x3FFBF5C1, 0x3FFC0B97, 0x3FFC216E,
  0x3FFC3748, 0x3FFC4D23, 0x3FFC6300, 0x3FFC78DF,
  0x3FFC8EC0, 0x3FFCA4A3, 0x3FFCBA88, 0x3FFCD06E,
  0x3FFCE657, 0x3FFCFC41, 0x3FFD122E, 0x3FFD281C,
  0x3FFD3E0C, 0x3FFD53FE, 0x3FFD69F2, 0x3FFD7FE8,
  0x3FFD95E0, 0x3FFDABD9, 0x3FFDC1D5, 0x3FFDD7D2,
  0x3FFDEDD2, 0x3FFE03D3, 0x3FFE19D6, 0x3FFE2FDB,
  0x3FFE45E2, 0x3FFE5BEB, 0x3FFE71F6, 0x3FFE8803,
  0x3FFE9E11, 0x3FFEB422, 0x3FFECA34, 0x3FFEE049,
  0x3FFEF65F, 0x3FFF0C77, 0x3FFF2291, 0x3FFF38AD,
  0x3FFF4ECB, 0x3FFF64EB, 0x3FFF7B0D, 0x3FFF9131,
  0x3FFFA756, 0x3FFFBD7E, 0x3FFFD3A7, 0x3FFFE9D3,
};

void xnn_math_f32_sigmoid__scalar_lut2048_p1_div(
    size_t n,
    const float* input,
    float* output)
{
  assert(n % sizeof(float) == 0);

  const float vmagic_bias = 0x1.800000p23f;
  // The largest z for which sigmoidf(-z) is normalized.
  // This number is also the largest z for which expf(-z) is normalized.
  const float vdenorm_cutoff = 0x1.5D589Ep+6f;
  const float vminus_log2e_x2048 = -0x1.715476p11f;
  // Last 18 bits are zeroes
  const float vln2_o2048_hi = 0x1.600000p-12f;
  const float vln2_o2048_lo = 0x1.7217F8p-19f;
  const float vone = 1.0f;

  const float vc1 = -0x1.FFFFFEp-1f;

  const uint32_t vindex_mask = UINT32_C(0x7FF);

  for (; n != 0; n -= sizeof(float)) {
    const float vx = *input++;

    // General structure of the algorithm:
    //           / exp(x) / (1 + exp(x)) if x <= 0
    //   f[x] := 
    //           \ 1 - f[-x] if x >= 0
    //
    // First we compute f[-z] := exp(-z) / (1 + exp(-z)) where z = abs(x),
    // then replace result with 1 - f[-z] if x >= 0.
    const float vz = fabsf(vx);

    // Compute reduced argument n := round(-z * 2048 / log(2)).
    // We do it by adding a large number (magic bias), which cause rounding of the result to an integer, then subtracing
    // the large number back. The first addition is combined with multiplication by log2e into a single FMA instruction.
    // The trick with adding large number is valid only within certain bounds (|z * 2048 / log(2)| <= 2**22, i.e.
    // |z| <= 0x1.62E43p+10 = 1419.5654296875), but that is acceptable, because inputs x outside of
    // [-87.336544, 17.328678] (i.e. z outsize [0, 87.336544]) underflow or saturate sigmoidf(x). We fixup the result
    // for such inputs at the very end of the algorithm.
    float vn = vz * vminus_log2e_x2048 + vmagic_bias;

    // Create a floating-point number s (scale) such that s := 2**(n / 2048) for such inputs that sigmoidf(-z) is
    // normalized, i.e. 0 <= z <= 87.33642. As n has 11 fractional bits, we split s == 2**(n / 2048) =
    // = 2**e * 2**(n / 2048 - e), where e := int(n / 2048). We create s in two steps:
    // 1. Fetch 2**(n / 2048 - e) = 2**(n % 2048) from exp2_k_over_2048_table using the 6 low bits of n, as integer.
    //    Note that the fetched values are in the [1.0, 2.0) range, i.e. their floating-point exponent is 0.
    // 2. Adjust fecthed value by addition of e to its floating-point exponent. The result is always a normalized
    //    number, because for 0 <= z <= 87.33642 (inputs for which sigmoidf(-z) is normalized) we have -126 <= e <= 0,
    //    and thus the adjusted exponent is not lower than -126.
    //
    // Extract e from bits 11:19 of n and shift it into bits 23:31 (position of floating-point exponent).
    const uint32_t ve = (fp32_to_bits(vn) & ~vindex_mask) << 12;

    // Use bits 0:11 bits of n, as integer, as an index for table lookup of l := 2**(n % 2048).
    const uint32_t vidx = fp32_to_bits(vn) & vindex_mask;
    // Adjust exponent of the value l fetched from the exp2_k_over_2048_table to get the final s value.
    const float vs = fp32_from_bits(exp2_k_over_2048_table[vidx] + ve);

    // Subtract the large number back to get the final n := round(-z * 2048 / log(2)) as a floating-point number.
    vn -= vmagic_bias;

    // Compute reduced argument t := (z + n * log(2) / 2048). Note that -t = -z - n * log(2) / 2048.
    // Use Cody-Waite range reduction method (note two constants to represent log(2) / 2048) to improve accuracy.
    float vt = vn * vln2_o2048_hi + vz;
    vt = vn * vln2_o2048_lo + vt;

    // Compute degree-1 polynomial approximation for exp(-t) on [-log(2)/4096, log(2)/4096]:
    //   P1(t) = 1 + t * c1
    const float vp = vt * vc1;

    // Reconstruct the exp(-z) value:
    //   y = s * (1 + t * c1)
    //     = s + s * (t * c1))
    //     = s + s * p
    const float vy = vp * vs + vs;

    // Reconstruct sigmoid(-z) = exp(-z) / (1.0 + exp(-z))
    float vf = vy / (vy + vone);

    // For inputs below denormal cutoff, replace output with +0.0f.
    // Note that for NaN inputs, comparison result is false, and outputs are left unchanged.
    if XNN_UNPREDICTABLE(vz > vdenorm_cutoff) {
      vf = 0.0f;
    }

    // Reconstruct sigmoid(x) = x < 0 ? sigmoid(-z) : 1.0 - sigmoid(-z)
    if XNN_UNPREDICTABLE(vx > 0.0f) {
      vf = vone - vf;
    }

    *output++ = vf;
  }
}
