/**********************************************************************
 * $my_monitor example -- C source code using TF/ACC PLI routines
 *
 * C source to:
 *   1. Add VCL flags to signals listed in the system task arguments.
 *   2. Whenever a signal changes value, print the current simulation
 *      time, old logic value and new logic value.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage: $show_all_nets(<module_instance>);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_MyMonitor_checktf(),
 *              PLIbook_MyMonitor_calltf();
 *  /* table entries -/
 *   {usertask,                       /* type of PLI routine -/
 *     0,                             /* user_data value -/
 *     PLIbook_MyMonitor_checktf,     /* checktf routine -/
 *     0,                             /* sizetf routine -/
 *     PLIbook_MyMonitor_calltf,      /* calltf routine -/
 *     0,                             /* misctf routine -/
 *     "$my_monitor",                 /* system task/function name -/
 *     1                              /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include "acc_user.h"         /* IEEE 1364 PLI ACC routine library */

/* prototype of consumer routine */
int PLIbook_MyMonitor_consumer();

/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_MyMonitor_checktf()
{
  int    i;
  static int validTypes[53] = {accNet, accNetBit, accReg, accRegBit, 0};

  acc_initialize();
  for (i=1; i<=tf_nump(); i++) {
    if ( !(acc_object_in_typelist(acc_handle_tfarg(i), validTypes)) )
      tf_error("$my_monitor argument %d must be a net or reg", i);
    else if ( tf_sizep(i) != 1)
      tf_error("$my_monitor argument %d must be scalar", i);
  }
  acc_close();
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
typedef struct PLIbook_MyMon_t {
  char   signalName[256];  /* signal names--up to 255 characters */
  char   lastValue[2];     /* scalar logic value stored as a string */
} PLIbook_MyMon_s, *PLIbook_MyMon_p;

PLIbook_MyMonitor_calltf() {
  handle signal_h;
  int    i, numargs = tf_nump();

  /* allocate memory for an array of p_monitor structures */
  PLIbook_MyMon_p monArray; /* starting address for the array */
  monArray = (PLIbook_MyMon_p)malloc(numargs*(sizeof(PLIbook_MyMon_s)));

  acc_initialize();
  /* save name and current logic value of each signal */
  for (i=0; i<numargs; i++) {
    signal_h = acc_handle_tfarg(i+1);
    strcpy(monArray[i].signalName, acc_fetch_fullname(signal_h));
    strcpy(monArray[i].lastValue, acc_fetch_value(signal_h, "%b", null));
  /* add a VCL flag to each net--user_data is a pointer to saved info */
    acc_vcl_add(signal_h, 
                PLIbook_MyMonitor_consumer,
                (char*)&monArray[i], 
                vcl_verilog_logic);
  }
  acc_close();
  return(0);
}
/**********************************************************************
 * consumer routine
 *********************************************************************/
int PLIbook_MyMonitor_consumer(p_vc_record vc_record)
{
  char newValue[2];
  /* retrieve pointer to data structure array from user_data field */
  PLIbook_MyMon_p ArrayElem_p = (PLIbook_MyMon_p)vc_record->user_data;
  switch (vc_record->vc_reason) { /* check reason call-back occurred */
    case logic_value_change:         /* scalar net changed */
    case sregister_value_change : {  /* scalar register changed */
      switch (vc_record->out_value.logic_value) { /* convert value */
        case vcl0: strcpy(newValue, "0"); break;  /* to string */
        case vcl1: strcpy(newValue, "1"); break;
        case vclX: strcpy(newValue, "x"); break;
        case vclZ: strcpy(newValue, "z"); break;
      }
      io_printf("At time %4d: %-20s last value=%s   new value=%s\n",
                vc_record->vc_lowtime, ArrayElem_p->signalName,
                ArrayElem_p->lastValue, newValue);
      strcpy(ArrayElem_p->lastValue, newValue); /* save the new value */
    }
  }
  return(0);
}
/*********************************************************************/

