/* 
 * File: wejpvolume.c
 *
 * A gtk2 volume control widget
 * Copyright (c) 2006 Johannes Heimansberg
 *
 * Released under the GNU General Public License v2
 */

#include "wejpvolume.h"
#include <string.h>
#include "volume0.xpm"
#include "volume1.xpm"
#include "volume2.xpm"
#include "volume3.xpm"

enum {
  VOLUME_CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint wejpvolume_signals[LAST_SIGNAL] = { 0 };

static GdkPixbuf *get_current_volume_icon(WejpVolume *wv)
{
	gint id = wv->current_volume / (wv->max_volume / 3);
	return wv->icon_volume[id];
}

GtkWidget *wejpvolume_new(guint max_volume,
                          guint initial_volume, 
                          gint  volume_step)
{
	GtkWidget *wv = GTK_WIDGET(g_object_new(WEJPVOLUME_TYPE, NULL));
	WEJPVOLUME(wv)->max_volume = max_volume;
	WEJPVOLUME(wv)->current_volume = initial_volume;
	WEJPVOLUME(wv)->volume_step = volume_step;

	WEJPVOLUME(wv)->icon_volume[0] = 
		gdk_pixbuf_new_from_xpm_data((const char **)volume0_xpm);
	WEJPVOLUME(wv)->icon_volume[1] = 
		gdk_pixbuf_new_from_xpm_data((const char **)volume1_xpm);
	WEJPVOLUME(wv)->icon_volume[2] = 
		gdk_pixbuf_new_from_xpm_data((const char **)volume2_xpm);
	WEJPVOLUME(wv)->icon_volume[3] = 
		gdk_pixbuf_new_from_xpm_data((const char **)volume3_xpm);
	WEJPVOLUME(wv)->image_status = gtk_image_new_from_pixbuf(get_current_volume_icon(WEJPVOLUME(wv)));

	gtk_container_add(GTK_CONTAINER(wv), WEJPVOLUME(wv)->image_status);

	gtk_widget_show_all(wv);

	return wv;
}

void wejpvolume_set_volume(WejpVolume *wv, guint volume)
{
	gint vol = (volume <= wv->max_volume ? volume : wv->max_volume);
	if (vol < 0) vol = 0;
	wv->current_volume = vol;
	gtk_image_set_from_pixbuf(GTK_IMAGE(wv->image_status),
	                          get_current_volume_icon(wv));
}

guint wejpvolume_get_volume(WejpVolume *wv)
{
	return wv->current_volume;
}

static void wejpvolume_class_init(WejpVolumeClass *class)
{
	wejpvolume_signals[VOLUME_CHANGED_SIGNAL] = 
		g_signal_new("volume-changed",
	                 G_TYPE_FROM_CLASS(class),
	                 G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
	                 G_STRUCT_OFFSET(WejpVolumeClass, wejpvolume),
                     NULL, 
                     NULL,                
	                 g_cclosure_marshal_VOID__VOID,
	                 G_TYPE_NONE,
	                 0);
}

void wejpvolume_draw(WejpVolume *wv)
{
}

static void cb_volume_change_click(GtkWidget      *widget,
                                   GdkEventButton *eb,
                                   gpointer        data)
{
	gint step = WEJPVOLUME(widget)->volume_step;

	switch (eb->button) {
		case 1:
			WEJPVOLUME(widget)->current_volume = 
				(WEJPVOLUME(widget)->current_volume + step <= WEJPVOLUME(widget)->max_volume ? 
				 WEJPVOLUME(widget)->current_volume + step : WEJPVOLUME(widget)->max_volume);
			break;
		case 3:
			WEJPVOLUME(widget)->current_volume = 
				(WEJPVOLUME(widget)->current_volume - step >= 0 ? 
				 WEJPVOLUME(widget)->current_volume - step : 0);
			break;
	}
	gtk_image_set_from_pixbuf(GTK_IMAGE(WEJPVOLUME(widget)->image_status),
	                          get_current_volume_icon(WEJPVOLUME(widget)));
	g_signal_emit(G_OBJECT(widget), wejpvolume_signals[VOLUME_CHANGED_SIGNAL], 0);
}

static void cb_volume_change_scroll(GtkWidget      *widget,
                                    GdkEventScroll *es,
                                    gpointer        data)
{
	gint step = WEJPVOLUME(widget)->volume_step;

	if (es->direction == GDK_SCROLL_UP) {
			WEJPVOLUME(widget)->current_volume = 
				(WEJPVOLUME(widget)->current_volume + step <= WEJPVOLUME(widget)->max_volume ? 
				 WEJPVOLUME(widget)->current_volume + step : WEJPVOLUME(widget)->max_volume);

	} else if (es->direction == GDK_SCROLL_DOWN) {
			WEJPVOLUME(widget)->current_volume = 
				(WEJPVOLUME(widget)->current_volume - step >= 0 ? 
				 WEJPVOLUME(widget)->current_volume - step : 0);
	}
	gtk_image_set_from_pixbuf(GTK_IMAGE(WEJPVOLUME(widget)->image_status),
	                          get_current_volume_icon(WEJPVOLUME(widget)));
	g_signal_emit(G_OBJECT(widget), wejpvolume_signals[VOLUME_CHANGED_SIGNAL], 0);
}

static void wejpvolume_init(WejpVolume *wv)
{
	g_signal_connect(G_OBJECT(wv), "button-press-event",
	                 G_CALLBACK(cb_volume_change_click), NULL);
	g_signal_connect(G_OBJECT(wv), "scroll-event",
	                 G_CALLBACK(cb_volume_change_scroll), NULL);
}

GType wejpvolume_get_type(void)
{
	static GType wv_type = 0;

	if (!wv_type)
	{
		static const GTypeInfo wv_info =
		{
			sizeof(WejpVolumeClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) wejpvolume_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof(WejpVolume),
			0,    /* n_preallocs */
			(GInstanceInitFunc) wejpvolume_init,
		};

		wv_type = g_type_register_static(GTK_TYPE_EVENT_BOX,
		                                 "WejpVolume",
		                                 &wv_info,
		                                 0);
	}
	return wv_type;
}
