// Copyright 2018 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package x509

import (
	"math/big"
	"testing"
)

func TestSECP192R1(t *testing.T) {
	curve := secp192r1()
	vectors := []struct {
		k    string // base 10
		x, y string // base 16
	}{
		// Test vectors from http://point-at-infinity.org/ecc/nisttv
		{
			k: "1",
			x: "188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012",
			y: "07192B95FFC8DA78631011ED6B24CDD573F977A11E794811",
		},
		{
			k: "2",
			x: "DAFEBF5828783F2AD35534631588A3F629A70FB16982A888",
			y: "DD6BDA0D993DA0FA46B27BBC141B868F59331AFA5C7E93AB",
		},
		{
			k: "3",
			x: "76E32A2557599E6EDCD283201FB2B9AADFD0D359CBB263DA",
			y: "782C37E372BA4520AA62E0FED121D49EF3B543660CFD05FD",
		},
		{
			k: "4",
			x: "35433907297CC378B0015703374729D7A4FE46647084E4BA",
			y: "A2649984F2135C301EA3ACB0776CD4F125389B311DB3BE32",
		},
		{
			k: "5",
			x: "10BB8E9840049B183E078D9C300E1605590118EBDD7FF590",
			y: "31361008476F917BADC9F836E62762BE312B72543CCEAEA1",
		},
		{
			k: "6",
			x: "A37ABC6C431F9AC398BF5BD1AA6678320ACE8ECB93D23F2A",
			y: "851B3CAEC99908DBFED7040A1BBDA90E081F7C5710BC68F0",
		},
		{
			k: "7",
			x: "8DA75A1F75DDCD7660F923243060EDCE5DE37F007011FCFD",
			y: "57CB5FCF6860B35418240DB8FDB3C01DD4B702F96409FFB5",
		},
		{
			k: "8",
			x: "2FA1F92D1ECCE92014771993CC14899D4B5977883397EDDE",
			y: "A338AFDEF78B7214273B8B5978EF733FF2DD8A8E9738F6C0",
		},
		{
			k: "9",
			x: "818A4D308B1CABB74E9E8F2BA8D27C9E1D9D375AB980388F",
			y: "01D1AA5E208D87CD7C292F7CBB457CDF30EA542176C8E739",
		},
		{
			k: "10",
			x: "AA7C4F9EF99E3E96D1AEDE2BD9238842859BB150D1FE9D85",
			y: "3212A36547EDC62901EE3658B2F4859460EB5EB2491397B0",
		},
		{
			k: "11",
			x: "1C995995EB76324F1844F7164D22B652280940370628A2AA",
			y: "EF1765CE37E9EB73029F556400FA77BDB34CB8611AAA9C04",
		},
		{
			k: "12",
			x: "1061343F3D456D0ECA013877F8C9E7B28FCCDCDA67EEB8AB",
			y: "5A064CAA2EA6B03798FEF8E3E7A48648681EAC020B27293F",
		},
		{
			k: "13",
			x: "112AF141D33EFB9F2F68821E051E4EA004144A363C4A090A",
			y: "6E0CBE3BFC5293F72A2C1726E081E09E7F10A094432B1C1E",
		},
		{
			k: "14",
			x: "13B9310646EBC93B591746B3F7C64E05DEE08843DE1081C1",
			y: "1EDCEA63B44142DD15F3B427EC41A1EC4FBACA95E186E6B4",
		},
		{
			k: "15",
			x: "8C9595E63B56B633BA3546B2B5414DE736DE4A9E7578B1E7",
			y: "266B762A934F00C17CF387993AA566B6AD7537CDD98FC7B1",
		},
		{
			k: "16",
			x: "B7310B4548FBFDBD29005092A5355BFCD99473733048AFDF",
			y: "FF9EAE9EDCD27C1E42D8585C4546D9491845C56629CF2290",
		},
		{
			k: "17",
			x: "44275CD2E1F46DC3F9F57636C2B4213B8BB445930510FF8A",
			y: "EFAD8348FDE30C87DE438612A818E98D9B76A67AD25DDFD0",
		},
		{
			k: "18",
			x: "C1B4DB0227210613A6CA15C428024E40B6513365D72591A3",
			y: "1E26B286BCA1D08F4FE8F801267DF9FD7782EC3EC3F47F53",
		},
		{
			k: "19",
			x: "C0626BCF247DE5D307FD839238D72688774FC97A1CF8AD1B",
			y: "9CDC99D753973DC197E12778E829C804EC1A6B4E71FAA20A",
		},
		{
			k: "20",
			x: "BB6F082321D34DBD786A1566915C6DD5EDF879AB0F5ADD67",
			y: "91E4DD8A77C4531C8B76DEF2E5339B5EB95D5D9479DF4C8D",
		},
		{
			k: "112233445566778899",
			x: "81E6E0F14C9302C8A8DCA8A038B73165E9687D0490CD9F85",
			y: "F58067119EED8579388C4281DC645A27DB7764750E812477",
		},
		{
			k: "112233445566778899112233445566778899",
			x: "B357B10AC985C891B29FB37DA56661CCCF50CEC21128D4F6",
			y: "BA20DC2FA1CC228D3C2D8B538C2177C2921884C6B7F0D96F",
		},
		{
			k: "1618292094200346491064154703205151664562462359653015613567",
			x: "74FEC215F253C6BD845831E059B318C87F727B136A700B91",
			y: "4B702B15B126A703E7A7CEC3E0EC81F8DFCA73A59F5D88B9",
		},
		{
			k: "1484605055214526729816930749766694384906446681761906688",
			x: "0C40230F9C4B8C0FD91F2C604FCBA9B87C2DFA153F010B4F",
			y: "5FC4F5771F467971B2C82752413833A68CE00F4A9A692B02",
		},
		{
			k: "1569275434166462877105627261392580354519833538813866540831",
			x: "28783BBF6208E1FF0F965FD8DC0C26FF1D8E02B433EDF2F7",
			y: "A5852BBC44FD8164C1ABA9A3EC7A88E461D5D77ABD743E87",
		},
		{
			k: "3138550867681922400546388175470823984762234518836963313664",
			x: "45DAF0A306121BDB3B82E734CB44FDF65C9930F0E4FD2068",
			y: "F039FACE58EB7DE34E3374ADB28DF81F019C4548BAA75B64",
		},
		{
			k: "3138550119404545973088374812479323842475901485681169401600",
			x: "1D5EC85004EA2ABA905CEF98A818A8C3516D7CB69A6FD575",
			y: "4008F35F5820F66C902195644162E5AA231DD69C9E1ECC97",
		},
		{
			k: "24519928471166604179655321383971467003990211439919824896",
			x: "F063727C2EA4D358AB02F6B0BEEB14DBEAF2E8A1DB3208EE",
			y: "427418C015553361769B6A0C42923C4CA103740B6DCD9703",
		},
		{
			k: "46756768218837031708063422466358611246556475572231",
			x: "DC81D33CA6604B1EFE49386CD492979EF807B8BAEB8566E3",
			y: "D454247FF478514556333B3901C9F1CCC18DBC9AB938CFA0",
		},
		{
			k: "3138502977207688322901699644928655553044791844086883549215",
			x: "D932741DF6AA0E1EED24279150436C752AA5ADCFD0698D72",
			y: "9759B6D2EF21D885E94CDFF219F17004D8763401DAB021B5",
		},
		{
			k: "47890485652059026491391979477371914515865621847605503",
			x: "571477E9D9F2A628780742257F7250C4224C483B30F3A97E",
			y: "1AD35EE3177D22DD5F01B5A46FFDEC547B6A41786EBB8C8F",
		},
		{
			k: "3138549376958826959341570842566593375326996431013993775615",
			x: "4C69939642792776C826DB8B4EBF4BD8C03FC9DFA2AEC822",
			y: "29BF35BE52A6036E07EBA5741CFEB4C143310216EF1B9A2E",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284061",
			x: "BB6F082321D34DBD786A1566915C6DD5EDF879AB0F5ADD67",
			y: "6E1B2275883BACE37489210D1ACC64A046A2A26B8620B372",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284062",
			x: "C0626BCF247DE5D307FD839238D72688774FC97A1CF8AD1B",
			y: "63236628AC68C23E681ED88717D637FA13E594B18E055DF5",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284063",
			x: "C1B4DB0227210613A6CA15C428024E40B6513365D72591A3",
			y: "E1D94D79435E2F70B01707FED9820601887D13C13C0B80AC",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284064",
			x: "44275CD2E1F46DC3F9F57636C2B4213B8BB445930510FF8A",
			y: "10527CB7021CF37821BC79ED57E71671648959852DA2202F",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284065",
			x: "B7310B4548FBFDBD29005092A5355BFCD99473733048AFDF",
			y: "00615161232D83E1BD27A7A3BAB926B5E7BA3A99D630DD6F",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284066",
			x: "8C9595E63B56B633BA3546B2B5414DE736DE4A9E7578B1E7",
			y: "D99489D56CB0FF3E830C7866C55A9948528AC8322670384E",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284067",
			x: "13B9310646EBC93B591746B3F7C64E05DEE08843DE1081C1",
			y: "E123159C4BBEBD22EA0C4BD813BE5E12B045356A1E79194B",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284068",
			x: "112AF141D33EFB9F2F68821E051E4EA004144A363C4A090A",
			y: "91F341C403AD6C08D5D3E8D91F7E1F6080EF5F6BBCD4E3E1",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284069",
			x: "1061343F3D456D0ECA013877F8C9E7B28FCCDCDA67EEB8AB",
			y: "A5F9B355D1594FC86701071C185B79B697E153FDF4D8D6C0",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284070",
			x: "1C995995EB76324F1844F7164D22B652280940370628A2AA",
			y: "10E89A31C816148CFD60AA9BFF0588414CB3479EE55563FB",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284071",
			x: "AA7C4F9EF99E3E96D1AEDE2BD9238842859BB150D1FE9D85",
			y: "CDED5C9AB81239D6FE11C9A74D0B7A6A9F14A14DB6EC684F",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284072",
			x: "818A4D308B1CABB74E9E8F2BA8D27C9E1D9D375AB980388F",
			y: "FE2E55A1DF72783283D6D08344BA831FCF15ABDE893718C6",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284073",
			x: "2FA1F92D1ECCE92014771993CC14899D4B5977883397EDDE",
			y: "5CC7502108748DEBD8C474A687108CBF0D22757168C7093F",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284074",
			x: "8DA75A1F75DDCD7660F923243060EDCE5DE37F007011FCFD",
			y: "A834A030979F4CABE7DBF247024C3FE12B48FD069BF6004A",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284075",
			x: "A37ABC6C431F9AC398BF5BD1AA6678320ACE8ECB93D23F2A",
			y: "7AE4C3513666F7240128FBF5E44256F0F7E083A8EF43970F",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284076",
			x: "10BB8E9840049B183E078D9C300E1605590118EBDD7FF590",
			y: "CEC9EFF7B8906E84523607C919D89D40CED48DABC331515E",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284077",
			x: "35433907297CC378B0015703374729D7A4FE46647084E4BA",
			y: "5D9B667B0DECA3CFE15C534F88932B0DDAC764CEE24C41CD",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284078",
			x: "76E32A2557599E6EDCD283201FB2B9AADFD0D359CBB263DA",
			y: "87D3C81C8D45BADF559D1F012EDE2B600C4ABC99F302FA02",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284079",
			x: "DAFEBF5828783F2AD35534631588A3F629A70FB16982A888",
			y: "229425F266C25F05B94D8443EBE4796FA6CCE505A3816C54",
		},
		{
			k: "6277101735386680763835789423176059013767194773182842284080",
			x: "188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012",
			y: "F8E6D46A003725879CEFEE1294DB32298C06885EE186B7EE",
		},
	}
	for _, test := range vectors {
		t.Run(test.k, func(t *testing.T) {
			k, ok := new(big.Int).SetString(test.k, 10)
			if !ok {
				t.Fatalf("failed to parse k=%s", test.k)
			}
			wantX, ok := new(big.Int).SetString(test.x, 16)
			if !ok {
				t.Fatalf("failed to parse x=%s", test.x)
			}
			wantY, ok := new(big.Int).SetString(test.y, 16)
			if !ok {
				t.Fatalf("failed to parse y=%s", test.y)
			}

			gotX, gotY := curve.ScalarBaseMult(k.Bytes())
			if gotX.Cmp(wantX) != 0 || gotY.Cmp(wantY) != 0 {
				t.Errorf("ScalarBaseMult()=(%v,%v), want (%v,%v)", gotX, gotY, wantX, wantY)
			}
		})
	}
}
