/***************************************************************************
 *   Copyright (C) 2004, 2005, 2006 by Stephen McInerney                   *
 *   spm@stedee.id.au                                                      *
 *                                                                         *
 *   $Id: options.c 66 2006-06-24 23:50:26Z steve $
 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/***************************************************************************
 ***************************************************************************
 * options.c
 *
 * Deal with Command line options
 ***************************************************************************
 ***************************************************************************/

/***************************************************************************
 ***************************************************************************
 * ModificationHistory:
 **********************
 * 17-Mar-2005 steve    Initial Creation - taken from visitors/options.c
 * 08-Oct-2005 steve    Add Individual IP History lookups
 *
 ***************************************************************************
 ***************************************************************************/

#include "dnshistory.h"

/************************************************************************
 *                              GLOBALS                                 *
 ************************************************************************/
char *g_filename = NULL;                        /* Command Line specified file to open. */
char *g_recombine_filename = NULL;              /* Command Line specified file to open.
                                                   This file is the source of existing FQDN's
                                                   It should be a near exact copy of the STDIN or g_filename log file */
char *g_import_filename = NULL;                 /* Command Line specified file to open for Importing a dumped DB */

bool g_dolookups = true;                        /* Default. Do lookups of addresses
                                                   If False then will retrieve results from DB
                                                   and output substituted log lines */
bool g_dotranslate = false;                     /* Do Translations */
bool g_dorecombine = false;                     /* Do Recombining */
bool g_dumpdnsdb = false;                       /* Dump the DNS History Database to STDOUT */
bool g_showhistory = false;                     /* Given one or more IP's show their history */
bool g_doimport = false;                        /* Import a Dumped DB from a File */

u_int32_t db_cache_size = DB_CACHE_SIZE;        /* Size of the DB Cache */
int g_dns_timeout = DNS_RESULTS_RECHECK;        /* The period that DNS queries won't be retried within. Seconds */
int g_dns_lookups = DNS_LOOKUPS;                /* Number of attempts for each lookup if not successful */
unsigned int g_max_threads = THREAD_MAXIMUM_THREADS;    /* Maximum number of threads to spawn off */
int g_dns_retry_delay = DNS_RETRY_DELAY;        /* Delay time between dns lookup retries in a single run. Seconds */

/************************************************************************
 *                              FUNCTIONS                               *
 ************************************************************************/
void process_options(int, char *[]);            /* Process command line options */
void display_version(void);                     /* Display the Version Information */
void display_usage(int);                        /* Display the program Command Line Options, and exit */
void display_gnu_info(int);                     /* Display the GNU Disclaimer, and exit */


/************************************************************************
 * display_version                                                      *
 *                                                                      *
 * Display the Version Information                                      *
 ************************************************************************/
void
display_version()
{
    printf("%s\n", PACKAGE_STRING);
    printf("Copyright (C) %s Stephen McInerney\n\n", "2004, 2005, 2006");
}


/************************************************************************
 * display_usage                                                        *
 *                                                                      *
 * Display the Command Line Use and Exit with the passed exit_code      *
 ************************************************************************/
void
display_usage(int exit_code)
{
    display_version();
    printf("Usage: dnshistory [OPTION]...\n");
    printf("  -L, --dolookups            Do Lookups and store into DB (default)\n");
    printf("  -D, --dodump               Dump the history Database to STDOUT\n");
    printf("  -I, --doimport=FILE        Import a Dumped history from FILE\n");
    printf("  -T, --dotranslate          Retrieve lookups from DB and replace in output\n");
    printf("  -R, --dorecombine=FILE     Recombine wth the Names in this log file\n");
    printf("  -S, --showhistory          Given one or more IP's show their history\n");
    printf("\n");
    printf("  --logtype=LOGTYPE          Set the log type to use\n");
    printf("                               auto, clf or www, squid, ftp, iptables\n");
    printf("                               (default = auto)\n");
    printf("\n");
    printf("  -c, --cache=SIZE           Size of the In Memory Cache to create in Mb\n");
    printf("                               (default = %d)\n", DB_CACHE_SIZE);
    printf("  -d, --database=FILE        Database File to store results in\n");
    printf("                               (default = %s)\n", DATABASE);
    printf("  -f, --file=FILE            Log File to process\n");
    printf("                               if not set will use STDIN\n");
    printf("  -h, --help                 Display this message\n");
    printf("  -l, --maxlookups=NUMBER    Number of attempts to try queries\n");
    printf("                               (default = %d)\n", DNS_LOOKUPS);
    printf("  -m, --maxthreads=NUMBER    Maximum number of threads\n");
    printf("                               (default = %d)\n", THREAD_MAXIMUM_THREADS);
    printf("  -t, --timeout=SECONDS      Time interval to ignore doing further lookups.\n");
    printf("                               (default = %d)\n", DNS_RESULTS_RECHECK);
    printf("  -v, --verbose              Level of verbosity, Multiple v's will increase\n");
    printf("  -V, --version              Display the version information and exit\n");
    printf("  -w, --wait=SECONDS         Delay time between query retries in a single run\n");
    printf("                               (default = %d)\n", DNS_RETRY_DELAY);

    exit(exit_code);
}


/************************************************************************
 * display_gnu_info                                                     *
 *                                                                      *
 * Display the GNU standard disclaimer                                  *
 ************************************************************************/
void
display_gnu_info(int exit_code)
{
    display_version();
    printf("%s\n%s%s%s\n%s\n%s\n",
           "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A",
           "PARTICULAR PURPOSE.  You may redistribute copies of ", PACKAGE_NAME, " under the terms", "the GNU General Public License.",
           "For more information about these matters, see the file named COPYING.");
    exit(exit_code);
}


/************************************************************************
 * process_options                                                      *
 *                                                                      *
 * Process the individual commandline arguments                         *
 ************************************************************************/
void
process_options(int argc, char *argv[])
{
    int opt;
    int option_index = 0;
    char log_type[20];

    /* Set the various exclusive options used flags */
    bool optL = false;
    bool optD = false;
    bool optI = false;
    bool optR = false;
    bool optT = false;
    bool optS = false;

    const char *short_options = "c:d:Df:F:h:I:Ll:m:R:STt:vVw:?";
    const struct option long_options[] = {
        {"cache", 1, NULL, 'c'},
        {"database", 1, NULL, 'd'},
        {"dodump", 0, NULL, 'D'},
        {"file", 1, NULL, 'f'},
        {"help", 0, NULL, 'h'},
        {"doimport", 1, NULL, 'I'},
        {"dolookups", 0, NULL, 'L'},
        {"maxthreads", 1, NULL, 'm'},
        {"dorecombine", 1, NULL, 'R'},
        {"lookups", 1, NULL, 'l'},
        {"showhistory", 0, NULL, 'S'},
        {"dotranslate", 0, NULL, 'T'},
        {"timeout", 1, NULL, 't'},
        {"verbose", 0, NULL, 'v'},
        {"version", 0, NULL, 'V'},
        {"wait", 1, NULL, 'w'},
        {"logtype", 1, NULL, 0},
        {NULL, 0, NULL, 0}
    };

    /**********************************************************************/

    do {
        opt = getopt_long(argc, argv, short_options, long_options, &option_index);
        switch (opt) {
        case 0:
            /*
               printf ("option %s", long_options[option_index].name);
               if (optarg)
               printf (" with arg %s", optarg);
               printf ("\n");
             */
            if (strncmp("logtype", long_options[option_index].name, 7) == 0) {
                sscanf(optarg, "%s", log_type);
                if (strncmp(log_type, "auto", 4) == 0) {
                    g_log_type = LOG_AUTO;
                } else if (strncmp(log_type, "clf", 3) == 0) {
                    g_log_type = LOG_CLF;
                } else if (strncmp(log_type, "www", 3) == 0) {
                    g_log_type = LOG_CLF;
                } else if (strncmp(log_type, "ftp", 3) == 0) {
                    g_log_type = LOG_FTP;
                } else if (strncmp(log_type, "squid", 5) == 0) {
                    g_log_type = LOG_SQUID;
                } else if (strncmp(log_type, "iptables", 8) == 0) {
                    g_log_type = LOG_IPTABLES;
                } else {
                    ERRVPRINT(VERBOSE0, "Unknown Log Type: %s\n", log_type);
                    exit(1);
                }
            }
            break;
        case 'L':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = true;
                g_dotranslate = false;
                g_dorecombine = false;
                g_dumpdnsdb = false;
                g_showhistory = false;
                g_doimport = false;
                optL = true;
            }
            break;
        case 'D':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = false;
                g_dotranslate = false;
                g_dorecombine = false;
                g_dumpdnsdb = true;
                g_showhistory = false;
                g_doimport = false;
                optD = true;
            }
            break;
        case 'I':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = false;
                g_dotranslate = false;
                g_dorecombine = false;
                g_dumpdnsdb = false;
                g_showhistory = false;
                g_doimport = true;
                g_import_filename = optarg;
                optI = true;
            }
            break;
        case 'R':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = false;
                g_dotranslate = false;
                g_dorecombine = true;
                g_dumpdnsdb = false;
                g_showhistory = false;
                g_doimport = false;
                g_recombine_filename = optarg;
                optR = true;
            }
            break;
        case 'S':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = false;
                g_dotranslate = false;
                g_dorecombine = false;
                g_dumpdnsdb = false;
                g_showhistory = true;
                g_doimport = false;
                optS = true;
                /* getopt will exit with optind set to the index for the IP's to lookup */
            }
            break;
        case 'T':
            if (optL || optR || optD || optT || optS || optI) {
                ERRVPRINT(VERBOSE0, "%s\n\n", msg_F_incompatible_options);
                display_usage(V_EXIT_INVALID_PARAMS);
            } else {
                g_dolookups = false;
                g_dotranslate = true;
                g_dorecombine = false;
                g_dumpdnsdb = false;
                g_showhistory = false;
                g_doimport = false;
                optT = true;
            }
            break;
        case 'c':
            sscanf(optarg, "%d", &db_cache_size);
            if (db_cache_size <= 0) {
                ERRVPRINT(VERBOSE0, "Database Cache Size must be greater than 0\n%s", "");
                display_usage(V_EXIT_INVALID_PARAMS);
            }
            break;
            break;
        case 'd':
            /* g_db_dirfilename = optarg; */
            strncpy(g_db_dirfilename, optarg, strlen(optarg) + 1);
            break;
        case 'h':
            display_usage(V_EXIT_OK);
            break;
        case 'f':
            g_filename = optarg;
            break;
        case 'l':
            sscanf(optarg, "%d", &g_dns_lookups);
            if (g_dns_lookups <= 0) {
                ERRVPRINT(VERBOSE0, "Lookup attempts must be greater than 0\n%s", "");
                display_usage(V_EXIT_INVALID_PARAMS);
            }
            break;
        case 'm':
            sscanf(optarg, "%d", &g_max_threads);
            if (g_max_threads <= 0) {
                ERRVPRINT(VERBOSE0, "Maximum threads must be greater than 0\n%s", "");
                display_usage(V_EXIT_INVALID_PARAMS);
            }
            break;
        case 't':
            sscanf(optarg, "%d", &g_dns_timeout);
            if (g_dns_timeout <= 0) {
                ERRVPRINT(VERBOSE0, "DNS Timeout must be greater than 0\n%s", "");
                display_usage(V_EXIT_INVALID_PARAMS);
            }
            break;
        case 'V':
            display_gnu_info(V_EXIT_OK);
            break;
        case 'v':
            g_verbosity++;
            if (g_verbosity > VERBOSE_MAX) {
                g_verbosity = VERBOSE_MAX;
                ERRVPRINT(VERBOSE0, "%s\n", msg_W_exceeded_verbosity);
            }
            break;
        case 'w':
            sscanf(optarg, "%d", &g_dns_retry_delay);
            if (g_dns_retry_delay <= 0) {
                ERRVPRINT(VERBOSE0, "DNS Rety Delay must be greater than 0\n%s", "");
                display_usage(V_EXIT_INVALID_PARAMS);
            }
            break;
        case '?':
            display_usage(V_EXIT_OK);
            break;
        case -1:
            break;
        default:
            exit(V_EXIT_INVALID_PARAMS);
        }
    } while (opt != -1);
    VPRINT(VERBOSE2, "Commandline Settings:%s\n", "");
    VPRINT(VERBOSE2, "  DB Cache Size: %d Mb\n", db_cache_size);
    VPRINT(VERBOSE2, "  DB File:       %s\n", g_db_dirfilename);
    VPRINT(VERBOSE2, "  Log File:      %s\n", g_filename);
    VPRINT(VERBOSE2, "  DNS Lookups:   %d\n", g_dns_lookups);
    VPRINT(VERBOSE2, "  Max Threads:   %d\n", g_max_threads);
    VPRINT(VERBOSE2, "  DNS Timeout:   %d\n", g_dns_timeout);
    VPRINT(VERBOSE2, "  Retry Delay:   %d\n", g_dns_retry_delay);
    VPRINT(VERBOSE2, "  Verbosity:     %d\n", g_verbosity);
    VPRINT(VERBOSE2, "  Log Type:      %d\n", g_log_type);
}

/************************************************************************
 ************************************************************************
 *                      END OF FILE                                     *
 ************************************************************************
 ************************************************************************/
