/*
 * audio stream functions
 */

#include "ReadStreamA.h"
#include "avifmt.h"
#include "avm_creators.h"
#include "audiodecoder.h"
#include "avm_cpuinfo.h"
#include "utils.h"
#include "avm_output.h"
#include <string.h> // memcpy
#include <stdlib.h>
#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

AVM_BEGIN_NAMESPACE;

static const short mpeg_bitrates[6][16] =
{
    /* -== MPEG-1 ==- */
    /* Layer I	 */
    { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448 },
    /* Layer II  */
    { 0, 32, 48, 56, 64, 80,  96, 112, 128, 160, 192, 224, 256, 320, 384 },
    /* Layer III */
    { 0, 32, 40, 48, 56, 64,  80, 96, 112, 128, 160, 192, 224, 256, 320 },

    /* -== MPEG-2 LSF ==- */
    /* Layer I	       */
    { 0, 32, 48, 56, 64, 80,  96, 112, 128, 144, 160, 176, 192, 224, 256 },
    /* Layers II & III */
    { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160 },
    { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160 },
};

static const int mpeg_sample_rates[4][3] =
{
    { 44100, 48000, 32000 }, // MPEG1
    { 22050, 24000, 16000 }, // MPEG2
    { 0, 0, 0 },	     // ERR
    { 11025, 12000, 8000 }   // MPEG2.5
};

/*
bits name	       comments
--------------------------------------------------
12   sync	       0xFFF  (12 bit is 0 for mpeg2.5 !)
1    version	       1=mpeg1.0, 0=mpeg2.0
2    lay	       4-lay = layerI, II or III
1    error protection  0=yes, 1=no
4    bitrate_index     see table below
2    sampling_freq     see table below
1    padding
1    extension	       see table below
2    mode	       see table below
2    mode_ext	       used with "joint stereo" mode
1    copyright	       0=no 1=yes
1    original	       0=no 1=yes
2    emphasis	       see table below
--------------------------------------------------
*/

#define MPEG_ID_MASK	    0x00180000
#define MPEG_MPEG1	    0x00180000
#define MPEG_MPEG2	    0x00100000
#define MPEG_MPEG2_5	    0x00000000

#define MPEG_LAYER_MASK     0x00060000
#define MPEG_LAYER_III	    0x00020000
#define MPEG_LAYER_II	    0x00040000
#define MPEG_LAYER_I	    0x00060000
#define MPEG_PROTECTION     0x00010000
#define MPEG_BITRATE_MASK   0x0000F000
#define MPEG_FREQUENCY_MASK 0x00000C00
#define MPEG_PAD_MASK	    0x00000200
#define MPEG_PRIVATE_MASK   0x00000100
#define MPEG_MODE_MASK	    0x000000C0
#define MPEG_MODE_EXT_MASK  0x00000030
#define MPEG_COPYRIGHT_MASK 0x00000008
#define MPEG_HOME_MASK	    0x00000004
#define MPEG_EMPHASIS_MASK  0x00000003

#define LAYER_I_SAMPLES       384
#define LAYER_II_III_SAMPLES 1152


Mp3AudioInfo::Mp3AudioInfo() : frame_size(0) {}

// MP3 header needs 40 bytes
int Mp3AudioInfo::Init(const void* buf, int fast)
{
    header = avm_get_be32(buf);
    //AVM_WRITE("audio reader", "HDR %x    %.4s\n", header, (char*)&header);
    // CHECK if it's mpeg & Get the sampling frequency

    int frequency_index;
    if ((header & 0xffe00000) != 0xffe00000
	|| ((layer = 4 - ((header & MPEG_LAYER_MASK) >> 17)) > 3)
	|| ((mode = (MPEG_MODE) (3 - ((header >> 19) & 3))) == Mp3AudioInfo::ERR) // MPEG_ID_MASK
	|| ((frequency_index = (header & MPEG_FREQUENCY_MASK)) == MPEG_FREQUENCY_MASK))
	return 0; // not mp3 header

    sample_rate = mpeg_sample_rates[mode][frequency_index >> 10];
    //AVM_WRITE("audio reader", "HDR %x    m:%d  fi:%d	 sr:%d\n", header, mode, frequency_index, sample_rate);
    // Get stereo mode index
    stereo_mode = (STEREO_MODE)((header & MPEG_MODE_MASK) >> 6);
    num_channels = (stereo_mode == MODE_MONO) ? 1 : 2;
    // Get layer
    samples_per_frame = LAYER_I_SAMPLES;
    if (layer > 1)
	samples_per_frame = LAYER_II_III_SAMPLES;

    start_offset = 40;
    if (fast)
	return start_offset;

    if (mode == MPEG2)
	samples_per_frame /= 2;
    else if (mode == MPEG2_5)
	samples_per_frame /= 4;

    // Check for VBR stream (Xing header)
    int b = 4;
    if (header & MPEG_ID_MASK)
	b += (stereo_mode != MODE_MONO) ? 32 : 17; // mpeg1 mpeg2
    else
	b += (stereo_mode != MODE_MONO) ? 17 : 9; // mpeg2_5
    xing = (avm_get_le32((uint8_t*)buf + b) == mmioFOURCC('X', 'i', 'n', 'g'));
    bitrate = (xing) ? 0 : GetBitrate();

    //printf ("%d %d	l:%d  %d   ch:%d\n", sample_rate, bitrate, layer, samples_per_frame, num_channels);
    frame_size = GetFrameSize();
    if (xing)
	start_offset += frame_size + 4;

    return start_offset;
}
int Mp3AudioInfo::GetBitrate() const
{
    int bitrate_idx = (header & MPEG_BITRATE_MASK) >> 12;
    int layer_idx = layer + ((layer > 0 && mode == MPEG1) ? -1 : 2);
    return mpeg_bitrates[layer_idx][bitrate_idx] * 1000;
}
int Mp3AudioInfo::GetFrameSize() const
{
    int tmp_bitrate = GetBitrate();

    if (!tmp_bitrate)
	return 0;

    int ret = (layer == 1) ? 48 : 144;

    if (mode == MPEG2 || mode == MPEG2_5)
	ret /= 2;
    //ret /= 4;
    //AVM_WRITE("audio reader", "%d  %d  %d \n", ret, tmp_bitrate, sample_rate);
    ret = ret * tmp_bitrate / sample_rate;
    if (header & MPEG_PAD_MASK)
	ret++;

    return ret;
}
void Mp3AudioInfo::PrintHeader() const
{
    static const char sf[][8] = { "MPEG1", "MPEG2", "ERR", "MPEG2.5" };
    static const char sm[][13] = { "Stereo", "JointStereo", "DualChannel", "Mono" };
    if (frame_size)
	AVM_WRITE("audio reader", "%s Layer-%d %dHz %dkbps %s %s"
		  "(%d,%d,%d)\n",
		  sf[mode], layer, sample_rate, bitrate / 1000,
		  sm[stereo_mode], xing ? "Xing " : "",
		  samples_per_frame, start_offset, frame_size);
}

ReadStreamA::ReadStreamA(IMediaReadStream* stream)
: ReadStream(stream), m_pAudiodecoder(0), m_uiMinSize(0), m_uiSrcSize(0), m_uiRead(0), m_bIsMp3(false)
{
    // FIXME - asf reading needs to know also about VideoStreams
    // currently this isn't possible to use it this way
    // Flush();

    // fix incorrect header
    if (m_WF->wFormatTag == 0x55
	&& (m_WF->cbSize != 12
	    || m_uiFormatSize < sizeof(MPEGLAYER3WAVEFORMAT)))
    {
	if (m_uiFormatSize < sizeof(MPEGLAYER3WAVEFORMAT))
	    AVM_WRITE("audio reader", "WARNING: WAVEFORMAT header missing MP3 header extension for MP3 audio track (sz:%lu != %" PRIsz ")\n",
		      (unsigned long)m_uiFormatSize, sizeof(MPEGLAYER3WAVEFORMAT));
	else
	    AVM_WRITE("audio reader", "WARNING: WAVEFORMAT header missing MP3 header extension for MP3 audio track (cb:%lu != 12)\n",
		      (unsigned long)m_WF->cbSize);
	m_uiFormatSize = sizeof(MPEGLAYER3WAVEFORMAT);
	char* n = new char[m_uiFormatSize];
	if (n)
	{
	    memcpy(n, m_pFormat, sizeof(WAVEFORMATEX));
	    delete[] m_pFormat;
	    m_pFormat = n;
	}
	m_WF->cbSize = sizeof(MPEGLAYER3WAVEFORMAT) - sizeof(WAVEFORMATEX);//12
        // everything past WAVEFORMATEX must be kept in stream native format
	avm_set_le16(&m_MP3WF->wID, MPEGLAYER3_ID_MPEG);    		// These values based on an old
	avm_set_le32(&m_MP3WF->fdwFlags, MPEGLAYER3_FLAG_PADDING_OFF);	// Usenet post!!
	avm_set_le16(&m_MP3WF->nBlockSize, 0);
	avm_set_le16(&m_MP3WF->nFramesPerBlock, 1);
	avm_set_le16(&m_MP3WF->nCodecDelay, 1393);			// 0x571 for F-IIS MP3 Codec
    }
    m_uiBps = m_WF->nAvgBytesPerSec;
    m_uiSampleSize = m_WF->nBlockAlign;

    switch (m_WF->wFormatTag)
    {
    case 0x55:
    case 0x50:
	{
	    Mp3AudioInfo ai;
	    Mp3AudioInfo last;
	    char b[8];
	    size_t bf = 0;
	    size_t rb = 0;
	    int maxloop = 3000;
	    memset(&last, 0, sizeof(last));
	    while (ai.frame_size == 0)
	    {
		if (bf)
		{
		    memmove(b, b + 1, bf);
		    bf--; // next byte
		}
		size_t samp, bytes = 4 - bf;
		//printf("BYTES %" PRIsz " %" PRIsz "\n", bytes, bf);
                // SampleSize is 1
		if (ReadDirect(b + bf, bytes, bytes, samp, bytes) < 0)
		    //|| !bytes)
		    break;
		//printf("Mp3Header READ %" PRIsz "  %" PRIsz "\n", bytes, samp);
		bf += bytes;
		rb += bytes;
		//printf("RB %d  %d   0x%x\n", rb, bytes, *(int*)b);
		if (!ai.Init(b, true))
		    continue;

		//ai.PrintHeader();
		if (rem_size > 36 && rem_local >= 4)
		    ai.Init(&rem_buffer[rem_local - 4]); // full init
		else
		{
		    ai.xing = 0;
		    ai.bitrate = ai.GetBitrate();
		    ai.frame_size = ai.GetFrameSize();
		}
		// now we have found 4 bytes which might be mp3 header
		// let's do some more checking
		//printf("BFSCAN  %d   %d  %d\n", bf, rem_size, rem_local);
		if ((ai.frame_size - 4) == rem_size && rem_local >= 4)
		{
		    // heuristic 1:
		    // looks like we have found reasonable good mp3 header
		    // it's matching block boundary of the input frame
		    //printf("YES 1st. match\n");
		    break;
		}
		//printf("AAA  %d  %d\n", ai.frame_size, rem_size);
		if ((ai.frame_size - 4) < rem_size)
		{
		    // we could check if the next packet has the same
		    // characteristic
		    Mp3AudioInfo bi;
		    if (bi.Init(&rem_buffer[rem_local + ai.frame_size - 4], true)
			&& bi.sample_rate == ai.sample_rate
			&& bi.layer == ai.layer)
		    {
			//printf("YES 2nd. match\n");
			break;
		    }
		    ai.frame_size = 0;
		    continue; // failed - try to get next mp3 header
		}
		// this is unsolved if we get here with to short packet
		//ai.PrintHeader();
		if (maxloop--)
		{
		    //last.PrintHeader();
		    if (last.sample_rate == ai.sample_rate
			&& last.layer == ai.layer)
			break;
		    last = ai;
		    ai.frame_size = 0;
		}
		else
		    AVM_WRITE("audio reader", "failed to easily identify mp3 header! (%lu, %lu, %lu)\n",
			      (unsigned long)rem_size, (unsigned long)rem_local, (unsigned long)ai.frame_size);

		//int ss = 0; for (uint_t x = 0; x < bytes; x++) ss += b[x];
		//AVM_WRITE("audio reader", "HR  %d    lb %d   samp: %d    %d  sum  %d\n", hr, bytes, samp, bf, ss);
	    }
	    if (ai.frame_size)
	    {
		if (rem_local >= 4)
		{
		    // restore last mp3 header
		    rem_local -= 4; // so unread
		    rem_size += 4;
		    rb -= 4;
		}
		ai.PrintHeader();
		m_WF->nChannels = (uint16_t)ai.num_channels;
		m_WF->nSamplesPerSec = ai.sample_rate;
		m_WF->nBlockAlign = 1; // Windows likes that this way...
		//printf("BLOCKALIGN %d  %d\n", m_WF->nBlockAlign, ai.frame_size);
		m_uiSampleSize = ai.frame_size;

		if (ai.layer == 3)
		{
		    if (m_WF->wFormatTag != 0x55)
		    {
			AVM_WRITE("audio reader", "WARNING: fixing bad MP3 Layer3 tag 0x%x -> 0x55\n", m_WF->wFormatTag);
			m_WF->wFormatTag = 0x55; // for layer3
		    }// FIXME ai.xing
		    if (!ai.xing && stream->GetSampleSize())
		    {
			if (ai.bitrate)
			{
			    // hard to calculate for VBR - so check only CBR
			    int bt = m_uiBps - ai.bitrate / 8;
			    if (bt < 0) bt = -bt; //labs
			    // do bitrate fix - but only if the difference
			    // is huge
			    // I've seen stream where packets where
			    // 16kbps 24kbps 16kbps --> ~20kbps
			    if (bt > 1000)
			    {
				if (stream->FixAvgBytes(ai.bitrate / 8) == 0)
				{
				    AVM_WRITE("audio reader", "WARNING: fixing wrong avg bitrate %dB/s -> %dB/s\n", m_uiBps, ai.bitrate / 8);
				    m_uiBps = m_WF->nAvgBytesPerSec = ai.bitrate / 8;
				}
			    }
			}
			//printf("SAMELESIZE %d\n", stream->GetSampleSize());
			int bs = avm_get_le16(&m_MP3WF->nBlockSize) - ai.frame_size;
			if (bs < 0) bs = -bs;
			if (bs > 256)
			{
			    AVM_WRITE("audio reader", "WARNING: fixing bad MP3 block size %d -> %d\n",
				      avm_get_le16(&m_MP3WF->nBlockSize), ai.frame_size);
			    avm_set_le16(&m_MP3WF->nBlockSize, (uint16_t)ai.frame_size);
			}
		    }
		}
		else
		{
		    if (m_WF->wFormatTag != 0x50)
		    {
			AVM_WRITE("audio reader", "WARNING: fixing bad MP3 Layer2 tag 0x%x - should be 0x50\n", m_WF->wFormatTag);
			m_WF->wFormatTag = 0x50; // for layer2
		    }
		}
		if (rb > 0)
		    AVM_WRITE("audio reader", "junk size at the begining:  time:%.2fs  pos:%u (%" PRIsz "b)\n", GetTime(), GetPos(), rb);
	    }
	}
	if (m_WF->wFormatTag == 0x55)
	    m_bIsMp3 = true;
	break;
    case 0x2000:
	if (m_WF->nSamplesPerSec > 48000)
	    m_WF->nSamplesPerSec = 48000;
	break;
    }

    //printf("AFTERCONST pos: %d  l:%d	s:%d\n", m_uiLastPos, rem_local, rem_size);
}

ReadStreamA::~ReadStreamA()
{
    StopStreaming();
}

void ReadStreamA::Flush()
{
    ReadStream::Flush();
    if (m_pAudiodecoder)
	m_pAudiodecoder->Flush();
}

// FIXME - when m_pAudiodecoder is running we may supply real data
size_t ReadStreamA::GetAudioFormat(void* wf, size_t size) const
{
    //printf("DATE  %p	%d\n", wf, size);
    //for (int i = 0; i < m_format_size; i++)
    //	  printf("### %d  %x\n", i, ((unsigned char*)m_format)[i]);
    if (wf)
	memcpy(wf, m_pFormat, (size < m_uiFormatSize) ? size : m_uiFormatSize);
    return m_uiFormatSize;
}

size_t ReadStreamA::GetFrameSize() const
{
    return m_uiMinSize;
}

size_t ReadStreamA::GetOutputFormat(void* format, size_t size) const
{
    if (!m_pAudiodecoder)
	return 0;
    if (!format || size < sizeof(WAVEFORMATEX))
	return sizeof(WAVEFORMATEX);

    return m_pAudiodecoder->GetOutputFormat((WAVEFORMATEX*)format);
}

framepos_t ReadStreamA::GetPos() const
{
    //printf("READA %d\n", m_uiSampleSize);
    framepos_t fp = m_uiLastPos;
    if (m_uiSampleSize)
    {
	//printf("READAAAAA %d	 %d    %d\n", (int)m_uiLastPos, (int)rem_size, (int)m_uiRead);
	fp += (framepos_t)(((int)m_uiRead - (int)rem_size) / (int)m_uiSampleSize);
    }

    return fp;
}

double ReadStreamA::GetTime(framepos_t pos) const
{
    //AVM_WRITE("audio reader", "AGetTime: %d	last:%f   rem_size:%d	bps:%d\n", pos, m_dLastTime, rem_size, m_uiBps);
    if (pos == ERR)
    {
	// Compensate time which is in buffer
	double smtime = m_dLastTime;
	if (m_uiBps > 0)
	{
	    //AVM_WRITE("audio reader", "AudioGetTime remsize %f   time: %f  pos: %d\n", rem_size/(double)m_uiBps, smtime, pos);
	    // rem_size must be kept small otherwise there will be sync problems with VBR audio
	    smtime += (double)((int)m_uiRead - (int)rem_size) / (double)m_uiBps;
	    if (smtime < 0.)
		smtime = 0.;
	}
	//printf("ret %f ->  %f  %" PRIsz "  %" PRIsz "  (%f)\n", m_dLastTime, smtime, rem_size,
	//	 m_uiRead, (double)rem_size/(double)m_uiBps);
	return smtime;
    }
    return m_pStream->GetTime(pos);
}

bool ReadStreamA::IsStreaming() const
{
    return (m_pAudiodecoder != NULL);
}

int ReadStreamA::ReadFrames(void* buffer, size_t bufsize, size_t samples,
			    size_t& samples_read, size_t& bytes_read)
{
    if (!m_pAudiodecoder || !samples || bufsize < m_uiMinSize)
	return -1;

    bool bHadPacket = false;
    //printf("***ReadFrames***	MIN  %d   SRC %d    reml:%d  rems:%d	bs:%d\n", m_uiMinSize, srcsize, rem_local, rem_size, bufsize);
    if (0 && m_bIsMp3 && rem_size >= 4)
    {
	// for vbr audio keep buffer minimal for the good timing
	Mp3AudioInfo ai;
	if (ai.Init(&rem_buffer[rem_local], false))
	    m_uiSrcSize = ai.frame_size * 2;
	//printf("RESULT %d  %d\n", (int)m_uiSrcSize, (int)ai.frame_size);
    }

    //printf("REMSIZE s:%" PRIsz "   lp:%" PRIsz "	  lim:%" PRIsz "\n", rem_size, rem_local, rem_limit);
    while (rem_size < m_uiSrcSize)
    {
	//printf("SRCSIZE  %d  %d  %d	 sam:%d  loc:%d  lim:%d\n", (int)bufsize, (int)m_uiSrcSize, (int)rem_size, (int)samples, (int)rem_local, (int)rem_limit);
	//AVM_WRITE("audio reader", "***********convert   %d -> %d    (rem: %d)  %f\n", m_uiSrcSize, bufsize, rem_size);
	//AVM_WRITE("audio reader", "-----------bufsize: %d   rem_size: %d   srcsize: %d  samples: %d\n", bufsize, rem_size, m_uiSrcSize, samples);
	if (m_pPacket)
	{
	    // wrap the buffer around
	    if ((rem_local + m_uiSrcSize) >= rem_limit)
	    {
		//printf("MEMMOVE s:%" PRIsz "	 lp:%" PRIsz "\n", rem_size, rem_local);
		memmove(&rem_buffer[0], &rem_buffer[rem_local], rem_size);
		rem_local = 0;
	    }
	    bHadPacket = true;
	    //printf("PACKETSIZE %" PRIsz "\n", m_pPacket->size);
	    // add samples to internal buffer
	    rem_size += m_pPacket->Read(&rem_buffer[rem_local + rem_size], m_uiSrcSize - rem_size);
	    m_uiRead = m_pPacket->GetRead();
	    if (m_pPacket->GetSize() > 0)
		break;
	    // packet empty -> read next packet
	}
	if (!ReadPacket())
	{
	    m_uiRead = 0;
	    if (!rem_size)
		m_iEof++;
	    break;
	}
	//printf("-----SAMPLES	%d  %d\n", rem_size, p->size);
    }

    if (1 && m_bIsMp3)
    {
	while (rem_size > 4)
	{
	    Mp3AudioInfo ai;
	    // using AudioInfo is slower but also more safe...
	    // this will skip bad mp3 chunks for Layer-3 audio stream
	    if (ai.Init(&rem_buffer[rem_local], true) > 0
		&& ai.sample_rate == ((WAVEFORMATEX*)m_pFormat)->nSamplesPerSec
		&& ai.layer == 3)
		break;
	    //uint32_t t = avm_get_be32(rem_buffer + rem_local); printf("SKIP %d   %d	%x\n", rem_size, srcsize, t);
	    // check next byte
	    rem_local++;
	    rem_size--;
	}
    }

    //for (int i = 0; i < 32; i++)  printf(" %02x", (unsigned char) rem_buffer[rem_local + i]); printf("\n");
    //Mp3AudioInfo ai; ai.Init(rem_buffer + rem_local, false); printf("AI SIZE	%d\n", ai.frame_size);

    size_t ocnt = 0;
    size_t lr = 0;
    int hr = m_pAudiodecoder->Convert(&rem_buffer[rem_local],
				      (m_uiSrcSize < rem_size) ? m_uiSrcSize : rem_size,
				      (char*)buffer, bufsize, &lr, &ocnt);
    //int ds = open("/tmp/xyz", O_WRONLY | O_CREAT | O_APPEND, 0666); write(ds, &rem_buffer[rem_local], lr); close(ds);
    //int ws = open("/tmp/wav", O_WRONLY | O_CREAT | O_APPEND, 0666); write(ws, buffer, ocnt); close(ws);
    //printf("CONVSZE bytes:%d	samples:%d   r:%d  rem_size:%d/%d   bs:%d\n", (int)lr, (int)ocnt, hr, (int)rem_size, (int)m_uiSrcSize, (int)bufsize);
    if (hr >= 0 && (ocnt > 0 || lr > 0))
    {
	if (lr > rem_size)
	    lr = rem_size;
	rem_local += lr;
	rem_size -= lr;
    }
    else
    {
	//size_t ba = (m_uiSrcSize < rem_size) ? m_uiSrcSize : rem_size;
	size_t ba = ((WAVEFORMATEX*)m_pFormat)->nBlockAlign ? : 1;
	//printf("ERRORCONVSZE hr:%d  lr:%d   oc:%d   rs:%d  ba:%d\n", hr, (int)lr, (int)ocnt, (int)rem_size, (int)ba);
	if (rem_size > ba)
	{
	    rem_size -= ba;
	    rem_local += ba;
	}
	else if (!bHadPacket)
	    rem_size = 0;
	ocnt = 0;
    }

    //AVM_WRITE("audio reader", "locread:%6d outputsz:%7d convsz:%6d lr:%d  bs:%d\n", rem_local, ocnt, convsz, lr, rem_size);
#if 0
    unsigned char* p = (unsigned char*)buffer + convsz;
    for (int i = -32; i < 64; i++)
    {
	if (!(i % 8))
	    AVM_WRITE("audio reader", "\n%4d ", i);
	//AVM_WRITE("audio reader", "  0x%4.4x", abs(p[i] - p[i-2]) & 0xffff);
	AVM_WRITE("audio reader", "  0x%02X", p[i]);
    }
#endif

    //printf("PACKET %d  %d\n", bHadPacket, rem_size);
    //AVM_WRITE("audio reader", 1, "ReadStreamA::ReadFrames conv %d bytes to %d (%d)\n",
    //		(int)rem_local, (int)ocnt, (int)bufsize);

    bytes_read = ocnt;
    samples_read = bytes_read;
    if (m_uiSampleSize > 1)
	samples_read /= m_uiSampleSize;

    //AVM_WRITE("audio reader", 3, "ReadStreamA: new sample is %d\n", m_pStream->GetPos());
    return 0;
}

int ReadStreamA::SkipTo(double pos)
{
    //printf("SKIPPOS %f   %f  %d  bps:%d\n", pos, GetTime(), m_uiSampleSize, m_uiBps);
    if (!m_uiSampleSize)
	return 0; // FIXME: can't skip in VBR audio stream for now

    uint8_t buf[MINSIZE];
    const size_t max = (sizeof(buf) - PADDING) / m_uiSampleSize;
    while ((pos - GetTime()) > 0.001)
    {
	size_t samples_read, bytes_read;
	size_t r = 2;
	if (m_uiBps > 0)
	{
	    if ((r = size_t((pos - GetTime()) * m_uiBps) / m_uiSampleSize) == 0)
		break; // can't get any closer

	    if (r > max)
		r = max;
	}
	AVM_WRITE("audio reader", 3, "Skip blocks %d  %d  pos:%f   t:%f\n", (int)r, (int)m_uiSampleSize, pos, GetTime());
	if (ReadDirect(buf, r * m_uiSampleSize, r, samples_read, bytes_read) < 0
	    || !bytes_read)
	    break;
    }

    return 0;
}

int ReadStreamA::SeekTime(double timepos)
{
    double t = GetTime();
    if ((timepos - t) < 0.002 && (timepos - t) > -0.002 && timepos > 0.002)
    {
	//printf("---------SEEK DISCARD %f  %f\n", t, timepos);
	return 0;
    }
    return ReadStream::SeekTime(timepos);
}

int ReadStreamA::StartStreaming(const char* privname)
{
    if (m_pAudiodecoder)
	return 0; // already streaming

    m_pAudiodecoder = CreateDecoderAudio((WAVEFORMATEX*)m_pFormat, privname);

    if (!m_pAudiodecoder)
    {
	AVM_WRITE("audio reader", "Failed to initialize audio decoder for format 0x%x\n",
		  ((WAVEFORMATEX*)m_pFormat)->wFormatTag);
	return -1;
    }

    m_uiMinSize = m_pAudiodecoder->GetMinSize();
    if (m_uiMinSize < MINSIZE)
	m_uiMinSize = MINSIZE;

    m_uiSrcSize = m_pAudiodecoder->GetSrcSize(SRCSIZE);
    if (m_uiSrcSize < SRCSIZE)
	m_uiSrcSize = SRCSIZE;

    if (rem_limit < (MINSIZE + m_uiSrcSize)) {
	rem_limit = MINSIZE + m_uiSrcSize;
	rem_buffer.reserve(rem_limit + PADDING);
	memset(rem_buffer.begin(), 0, rem_limit + PADDING);
    }
    AVM_WRITE("audio reader", 1, "Initialize audio decoder - minsize: %" PRIsz " srsize: %" PRIsz "\n", m_uiMinSize, m_uiSrcSize);
    return 0;
}

int ReadStreamA::StopStreaming()
{
    if (m_pAudiodecoder)
    {
	FreeDecoderAudio(m_pAudiodecoder);
	m_pAudiodecoder = 0;
	m_uiMinSize = 0;
    }
    return 0;
}

AVM_END_NAMESPACE;
