/*
   A C-program for MT19937, with initialization improved 2002-01-26.
   Coded by Takuji Nishimura and Makoto Matsumoto.

   Before using, initialize the state by using mt_seed(seed)
   or mt_seed_array(init_key, key_length).

   Copyright (C) 1997-2002, Makoto Matsumoto and Takuji Nishimura,
   All rights reserved.
   Copyright (C) 2005-2007, Wessel Dankers.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote
        products derived from this software without specific prior written
        permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

   Any feedback is very welcome.
   http://www.math.keio.ac.jp/matumoto/emt.html
   email: matumoto@math.keio.ac.jp
*/

#include <stdint.h>

#include "mt19937ar.h"

/* Period parameters */
#define M 397
#define N 624
#define MT_UNINITIALIZED 625
#define MATRIX_A UINT32_C(0x9908b0df)   /* constant vector a */
#define UPPER_MASK UINT32_C(0x80000000) /* most significant w-r bits */
#define LOWER_MASK UINT32_C(0x7fffffff) /* least significant r bits */

static uint32_t mt[N];  /* the array for the state vector  */
static uint32_t mti = MT_UNINITIALIZED;

/* initializes mt[N] with a seed */
void mt_seed(uint32_t s) {
	/* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
	/* In the previous versions, MSBs of the seed affect   */
	/* only MSBs of the array mt.                          */
	for(mti = 1; mti < N; mti++)
		mt[mti] = UINT32_C(1812433253) * (mt[mti - 1] ^ (mt[mti - 1] >> 30)) + mti;
}

/* initialize by an array with array-length */
/* init_key is the array for initializing keys */
/* key_length is its length */
void mt_seed_array(uint32_t *init_key, unsigned int key_length) {
	unsigned int i = 1, j = 0, k;
	mt_seed(UINT32_C(19650218));
	for(k = key_length < N ? key_length : N; k; k--) {
		mt[i] = (mt[i] ^ ((mt[i - 1] ^ (mt[i - 1] >> 30))
			* UINT32_C(1664525))) + init_key[j] + (uint32_t)j; /* non linear */
		if(++i >= N) {
			mt[0] = mt[N - 1];
			i=1;
		}
		if(++j >= key_length)
			j=0;
	}
	for(k = N - 1; k; k--) {
		mt[i] = (mt[i] ^ ((mt[i - 1] ^ (mt[i - 1] >> 30))
			* UINT32_C(1566083941))) - (uint32_t)i; /* non linear */
		if(++i >= N) {
			mt[0] = mt[N - 1];
			i=1;
		}
	}

	/* MSB is 1; assuring non-zero initial array */
	mt[0] = UPPER_MASK;
}

/* generates a random number on [0,0xffffffff]-interval */
uint32_t mt_genrand32(void) {
	/* mag01[x] = x * MATRIX_A  for x = 0, 1 */
	static uint32_t mag01[2] = {UINT32_C(0), MATRIX_A};
	uint32_t y;

	if(mti >= N) { /* generate N words at one time */
		int kk;

		if(mti == MT_UNINITIALIZED) /* if init_genrand() has not been called, */
			mt_seed(UINT32_C(5489));   /* a default initial seed is used */

		for(kk = 0; kk < N - M; kk++) {
			y = (mt[kk] & UPPER_MASK) | (mt[kk + 1] & LOWER_MASK);
			mt[kk] = mt[kk + M] ^ (y >> 1) ^ mag01[y & UINT32_C(1)];
		}
		for(; kk < N-1; kk++) {
			y = (mt[kk] & UPPER_MASK) | (mt[kk + 1] & LOWER_MASK);
			mt[kk] = mt[kk + (M - N)] ^ (y >> 1) ^ mag01[y & UINT32_C(1)];
		}
		y = (mt[N - 1] & UPPER_MASK) | (mt[0] & LOWER_MASK);
		mt[N - 1] = mt[M - 1] ^ (y >> 1) ^ mag01[y & UINT32_C(1)];

		mti = 0;
	}

	y = mt[mti++];

	/* Tempering */
	y ^= y >> 11;
	y ^= (y << 7) & UINT32_C(0x9d2c5680);
	y ^= (y << 15) & UINT32_C(0xefc60000);
	y ^= y >> 18;

	return y;
}

/* generates a random number on [0,0x7fffffff]-interval */
int32_t mt_genrand31(void) {
	return (int32_t)(mt_genrand32() >> 1);
}

/* These real versions are due to Isaku Wada, 2002/01/09 added */
/* generates a random number on [0,1]-real-interval */
double mt_genrand_real1(void) {
	/* divided by 2^32-1 */
	uint32_t x = mt_genrand32();
	return (double)x * (1.0 / 4294967295.0);
}

/* generates a random number on [0,1)-real-interval */
double mt_genrand_real2(void) {
	/* divided by 2^32 */
	uint32_t x = mt_genrand32();
	return (double)x * (1.0 / 4294967296.0);
}

/* generates a random number on (0,1)-real-interval */
double mt_genrand_real3(void) {
	/* divided by 2^32 */
	uint32_t x = mt_genrand32();
	return ((double)x + 0.5) * (1.0 / 4294967296.0);
}

/* generates a random number on [0,1) with 53-bit resolution*/
double mt_genrand_res53(void) {
	uint32_t a, b;
	a = mt_genrand32() >> 5;
	b = mt_genrand32() >> 6;
	/* multiplied with 2^26, divided by 2^53 */
	return (a * 67108864.0 + b) * (1.0 / 9007199254740992.0);
}

uint32_t mt_genrand32_bounded(uint32_t min, uint32_t max) {
	uint64_t x = max - min;
	x *= (uint64_t)mt_genrand32();
	return min + (uint32_t)(x >> 32);
}
